//-----------------------------------------------------------------------------
// $Header:: 
// (C) Copyright 2001 NXP Semiconductors, All rights reserved
//
// This source code and any compilation or derivative thereof is the sole
// property of NXP Corporation and is provided pursuant to a Software
// License Agreement.  This code is the proprietary information of NXP
// Corporation and is confidential in nature.  Its use and dissemination by
// any party other than NXP Corporation is strictly limited by the
// confidential information provisions of the Agreement referenced above.
//-----------------------------------------------------------------------------
// FILE NAME:    tmbslTDA182I2.c
//
// %version: 25 %
//
// DESCRIPTION:  Function for the Hybrid silicon tuner TDA182I2
//
// DOCUMENT REF: 
//
// NOTES:
//-----------------------------------------------------------------------------
//

//-----------------------------------------------------------------------------
// Standard include files:
//-----------------------------------------------------------------------------
//

#include "tmNxTypes.h"
#include "tmCompId.h"
#include "tmFrontEnd.h"
#include "tmbslFrontEndTypes.h"

#include "tmddTDA182I2.h"
#define TMBSL_TDA18272

#ifdef TMBSL_TDA18272
 #include "tmbslTDA18272.h"
#else /* TMBSL_TDA18272 */
 #include "tmbslTDA18212.h"
#endif /* TMBSL_TDA18272 */

//-----------------------------------------------------------------------------
// Project include files:
//-----------------------------------------------------------------------------
//
#include "tmbslTDA182I2local.h"
#include "tmbslTDA182I2Instance.h"

//-----------------------------------------------------------------------------
// Types and defines:
//-----------------------------------------------------------------------------
//

//-----------------------------------------------------------------------------
// Global data:
//-----------------------------------------------------------------------------
//

//-----------------------------------------------------------------------------
// Exported functions:
//-----------------------------------------------------------------------------
//

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslTDA18211Init:
//
// DESCRIPTION: create an instance of a TDA182I2 Tuner
//
// RETURN:      TMBSL_ERR_TUNER_BAD_UNIT_NUMBER
//              TM_OK
//  
// NOTES:
//-----------------------------------------------------------------------------
//
tmErrorCode_t
tmbslTDA182I2Init
(
    tmUnitSelect_t              tUnit,      /* I: Unit number */
    tmbslFrontEndDependency_t*  psSrvFunc   /*  I: setup parameters */
)
{
    ptmTDA182I2Object_t pObj = Null;
    tmErrorCode_t       err  = TM_OK;

    if (psSrvFunc == Null)
    {
        err = TDA182I2_ERR_BAD_PARAMETER;
    }

    if(err == TM_OK)
    {
        //----------------------
        // initialize the Object
        //----------------------
        // pObj initialization
        err = TDA182I2GetInstance(tUnit, &pObj);
    }

    /* check driver state */
    if (err == TM_OK || err == TDA182I2_ERR_NOT_INITIALIZED)
    {
        if (pObj != Null && pObj->init == True)
        {
            err = TDA182I2_ERR_NOT_INITIALIZED;
        }
        else 
        {
            /* initialize the Object */
            if (pObj == Null)
            {
                err = TDA182I2AllocInstance(tUnit, &pObj);
                if (err != TM_OK || pObj == Null)
                {
                    err = TDA182I2_ERR_NOT_INITIALIZED;        
                }
            }

            if (err == TM_OK)
            {
                // initialize the Object by default values
                pObj->sRWFunc = psSrvFunc->sIo;
                pObj->sTime = psSrvFunc->sTime;
                pObj->sDebug = psSrvFunc->sDebug;

                if(  psSrvFunc->sMutex.Init != Null
                    && psSrvFunc->sMutex.DeInit != Null
                    && psSrvFunc->sMutex.Acquire != Null
                    && psSrvFunc->sMutex.Release != Null)
                {
                    pObj->sMutex = psSrvFunc->sMutex;

                    err = pObj->sMutex.Init(&pObj->pMutex);
                }

                pObj->init = True;
                err = TM_OK;

                err = tmddTDA182I2Init(tUnit, psSrvFunc);
                tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2Init(0x%08X) failed.", pObj->tUnit));
            }
        }
    }

    return err;
}

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslTDA182I2DeInit:
//
// DESCRIPTION: destroy an instance of a TDA182I2 Tuner
//
// RETURN:      TMBSL_ERR_TUNER_BAD_UNIT_NUMBER
//              TDA182I2_ERR_NOT_INITIALIZED
//              TM_OK
//
// NOTES:
//-----------------------------------------------------------------------------
//
tmErrorCode_t 
tmbslTDA182I2DeInit
(
    tmUnitSelect_t  tUnit   /* I: Unit number */
)
{
    ptmTDA182I2Object_t pObj = Null;
    tmErrorCode_t       err = TM_OK;

    /* check input parameters */
    err = TDA182I2GetInstance(tUnit, &pObj);
    tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "TDA182I2GetInstance(0x%08X) failed.", tUnit));

    if(err == TM_OK)
    {
        err = TDA182I2MutexAcquire(pObj, TDA182I2_MUTEX_TIMEOUT);
    }

    if(err == TM_OK)
    {
        err = tmddTDA182I2DeInit(tUnit);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2DeInit(0x%08X) failed.", pObj->tUnit));

        (void)TDA182I2MutexRelease(pObj);

        if(pObj->sMutex.DeInit != Null)
        {
            if(pObj->pMutex != Null)
            {
                err = pObj->sMutex.DeInit(pObj->pMutex);
            }

            pObj->sMutex.Init = Null;
            pObj->sMutex.DeInit = Null;
            pObj->sMutex.Acquire = Null;
            pObj->sMutex.Release = Null;

            pObj->pMutex = Null;
        }
    }

    err = TDA182I2DeAllocInstance(tUnit);

    return err;
}

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslTDA182I2GetSWVersion:
//
// DESCRIPTION: Return the version of this device
//
// RETURN:      TM_OK
//
// NOTES:       Values defined in the tmTDA182I2local.h file
//-----------------------------------------------------------------------------
//
tmErrorCode_t
tmbslTDA182I2GetSWVersion
(
    ptmSWVersion_t  pSWVersion  /* I: Receives SW Version */
)
{
    tmErrorCode_t   err = TDA182I2_ERR_NOT_INITIALIZED;
    
    err = tmddTDA182I2GetSWVersion(pSWVersion);
    
    return err;
}

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslTDA182I2CheckHWVersion:
//
// DESCRIPTION: Check HW version
//
// RETURN:      TM_OK if no error
//
// NOTES:       Values defined in the tmTDA182I2local.h file
//-----------------------------------------------------------------------------
tmErrorCode_t
tmbslTDA182I2CheckHWVersion
(
    tmUnitSelect_t tUnit    /* I: Unit number */
)
{
    ptmTDA182I2Object_t pObj = Null;
    tmErrorCode_t       err = TDA182I2_ERR_NOT_INITIALIZED;
    UInt16              uIdentity = 0;
    UInt8               majorRevision = 0;

    err = TDA182I2GetInstance(tUnit, &pObj);
    tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "TDA182I2GetInstance(0x%08X) failed.", tUnit));

    if(err == TM_OK)
    {
        err = TDA182I2MutexAcquire(pObj, TDA182I2_MUTEX_TIMEOUT);
    }

    if(err == TM_OK)
    {
        err = tmddTDA182I2GetIdentity(tUnit, &uIdentity);

        if(err == TM_OK)
        {
            if(uIdentity == 18272 || uIdentity == 18212)
            {
                /* TDA18272/12 found. Check Major Revision*/
                err = tmddTDA182I2GetMajorRevision(tUnit, &majorRevision);
                tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2GetMajorRevision(0x%08X) failed.", tUnit));

                if(err == TM_OK && majorRevision != 1)
                {
                    /* Only TDA18272/12 ES2 are supported */
                    err = TDA182I2_ERR_BAD_VERSION;
                }
            }
            else
            {
                err = TDA182I2_ERR_BAD_VERSION;
            }
        }

        (void)TDA182I2MutexRelease(pObj);
    }

    return err;
}

//-------------------------------------------------------------------------------------
// FUNCTION:    tmbslTDA182I2SetPowerState:
//
// DESCRIPTION: Set the power state of this device.
//
// RETURN:      TMBSL_ERR_TUNER_BAD_UNIT_NUMBER
//              TDA182I2_ERR_NOT_INITIALIZED
//              TM_OK
//
// NOTES:       
//-------------------------------------------------------------------------------------
//
tmErrorCode_t
tmbslTDA182I2SetPowerState
(
    tmUnitSelect_t          tUnit,      /* I: Unit number */
    tmTDA182I2PowerState_t  powerState  /* I: Power state of this device */
)
{
    ptmTDA182I2Object_t pObj = Null;
    tmErrorCode_t       err  = TM_OK;

    // pObj initialization
    err = TDA182I2GetInstance(tUnit, &pObj);
    tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "TDA182I2GetInstance(0x%08X) failed.", tUnit));

    if(err == TM_OK)
    {
        err = TDA182I2MutexAcquire(pObj, TDA182I2_MUTEX_TIMEOUT);
    }

    if(err == TM_OK)
    {
        // Call tmddTDA182I2SetPowerState
        err = tmddTDA182I2SetPowerState(tUnit, (tmddTDA182I2PowerState_t)powerState);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetPowerState(0x%08X) failed.", tUnit));

        if(err == TM_OK)
        {
            // set power state
            pObj->curPowerState = powerState;
        }

        (void)TDA182I2MutexRelease(pObj);
    }

    return err;
}

//-------------------------------------------------------------------------------------
// FUNCTION:    tmbslTDA182I2GetPowerState:
//
// DESCRIPTION: Get the power state of this device.
//
// RETURN:      TMBSL_ERR_TUNER_BAD_UNIT_NUMBER
//              TDA182I2_ERR_NOT_INITIALIZED
//              TM_OK
//
// NOTES:
//-------------------------------------------------------------------------------------
//
tmErrorCode_t
tmbslTDA182I2GetPowerState
(
    tmUnitSelect_t          tUnit,          /* I: Unit number */
    tmTDA182I2PowerState_t  *pPowerState    /* O: Power state of this device */
)
{
    ptmTDA182I2Object_t pObj = Null;
    tmErrorCode_t       err  = TM_OK;

    if(pPowerState == Null)
        err = TDA182I2_ERR_BAD_PARAMETER;

    if(err == TM_OK)
    {
        // pObj initialization
        err = TDA182I2GetInstance(tUnit, &pObj);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "TDA182I2GetInstance(0x%08X) failed.", tUnit));
    }

    if(err == TM_OK)
    {
        err = TDA182I2MutexAcquire(pObj, TDA182I2_MUTEX_TIMEOUT);
    }

    if(err == TM_OK)
    {
        // get power state
        *pPowerState = pObj->curPowerState;

        (void)TDA182I2MutexRelease(pObj);
    }

    return err;
}

//-------------------------------------------------------------------------------------
// FUNCTION:    tmbslTDA182I2SetStandardMode:
//
// DESCRIPTION: Set the standard mode of this device.
//
// RETURN:      TMBSL_ERR_TUNER_BAD_UNIT_NUMBER
//              TDA182I2_ERR_NOT_INITIALIZED
//              TM_OK
//
// NOTES:       
//-------------------------------------------------------------------------------------
//
tmErrorCode_t
tmbslTDA182I2SetStandardMode
(
    tmUnitSelect_t              tUnit,          /* I: Unit number */
    tmTDA182I2StandardMode_t    StandardMode    /* I: Standard mode of this device */
)
{
    ptmTDA182I2Object_t pObj = Null;
    tmErrorCode_t       err  = TM_OK;

    // pObj initialization
    err = TDA182I2GetInstance(tUnit, &pObj);
    tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "TDA182I2GetInstance(0x%08X) failed.", tUnit));

    if(err == TM_OK)
    {
        err = TDA182I2MutexAcquire(pObj, TDA182I2_MUTEX_TIMEOUT);
    }

    if(err == TM_OK)
    {
        // store standard mode 
        pObj->StandardMode = StandardMode;

        (void)TDA182I2MutexRelease(pObj);
    }

    return err;
}

//-------------------------------------------------------------------------------------
// FUNCTION:    tmbslTDA182I2GetStandardMode:
//
// DESCRIPTION: Get the standard mode of this device.
//
// RETURN:      TMBSL_ERR_TUNER_BAD_UNIT_NUMBER
//              TDA182I2_ERR_NOT_INITIALIZED
//              TM_OK
//
// NOTES:
//-------------------------------------------------------------------------------------
//
tmErrorCode_t
tmbslTDA182I2GetStandardMode
(
    tmUnitSelect_t              tUnit,          /* I: Unit number */
    tmTDA182I2StandardMode_t    *pStandardMode  /* O: Standard mode of this device */
)
{
    ptmTDA182I2Object_t pObj = Null;
    tmErrorCode_t       err  = TM_OK;

    if(pStandardMode == Null)
        err = TDA182I2_ERR_BAD_PARAMETER;

    if(err == TM_OK)
    {
        // pObj initialization
        err = TDA182I2GetInstance(tUnit, &pObj);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "TDA182I2GetInstance(0x%08X) failed.", tUnit));
    }

    if(err == TM_OK)
    {
        err = TDA182I2MutexAcquire(pObj, TDA182I2_MUTEX_TIMEOUT);
    }

    if(err == TM_OK)
    {
        /* Get standard mode */
        *pStandardMode = pObj->StandardMode;

        (void)TDA182I2MutexRelease(pObj);
    }

    return err;
}

//-------------------------------------------------------------------------------------
// FUNCTION:    tmbslTDA182I2SetRf:
//
// DESCRIPTION: Calculate i2c I2CMap & write in TDA182I2
//
// RETURN:      TMBSL_ERR_TUNER_BAD_UNIT_NUMBER
//              TDA182I2_ERR_NOT_INITIALIZED
//              TDA182I2_ERR_BAD_PARAMETER
//              TMBSL_ERR_IIC_ERR
//              TM_OK
//
// NOTES:       
//-------------------------------------------------------------------------------------
//
tmErrorCode_t
tmbslTDA182I2SetRf
(
    tmUnitSelect_t  tUnit,  /* I: Unit number */
    UInt32          uRF     /* I: RF frequency in hertz */
)
{    
    ptmTDA182I2Object_t pObj = Null;
    tmErrorCode_t       err  = TM_OK;
    Bool                bIRQWait = True;
    /* ---------- ES2 PATCH BEGIN ---------- */
    UInt8               majorRevision = 0;
    UInt8               minorRevision = 0;
    /* ---------- ES2 PATCH END ---------- */
    UInt8   ratioL, ratioH;  /* patch Spurious */
    UInt32  DeltaL, DeltaH; /* patch Spurious */

    //------------------------------
    // test input parameters
    //------------------------------
    // pObj initialization
    err = TDA182I2GetInstance(tUnit, &pObj);
    tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "TDA182I2GetInstance(0x%08X) failed.", tUnit));

    if(err == TM_OK)
    {
        err = TDA182I2MutexAcquire(pObj, TDA182I2_MUTEX_TIMEOUT);
    }

    if(err == TM_OK)
    {
        err = tmddTDA182I2GetIRQWait(tUnit, &bIRQWait);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2GetIRQWait(0x%08X) failed.", tUnit));

        pObj->uRF = uRF;

        if(err == TM_OK)
        {
            /* Set LPF */
            err = tmddTDA182I2SetLP_FC(tUnit, pObj->Std_Array[pObj->StandardMode].LPF);
            tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetLP_FC(0x%08X) failed.", tUnit));
        }
 
        if(err == TM_OK)
        {
            /* Set LPF Offset */
            err = tmddTDA182I2SetLP_FC_Offset(tUnit, pObj->Std_Array[pObj->StandardMode].LPF_Offset);
            tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetLP_FC_Offset(0x%08X) failed.", tUnit));
        }

        if(err == TM_OK)
        {
            /* Set IF Gain */
            err = tmddTDA182I2SetIF_Level(tUnit, pObj->Std_Array[pObj->StandardMode].IF_Gain);
            tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetIF_Level(0x%08X) failed.", tUnit));
        }

        if(err == TM_OK)
        {
            /* Set IF Notch */
            err = tmddTDA182I2SetIF_ATSC_Notch(tUnit, pObj->Std_Array[pObj->StandardMode].IF_Notch);
            tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetIF_ATSC_Notch(0x%08X) failed.", tUnit));
        }

        if(err == TM_OK)
        {
            /* Enable/disable HPF */
            if ( pObj->Std_Array[pObj->StandardMode].IF_HPF == tmTDA182I2_IF_HPF_Disabled )
            {
                err = tmddTDA182I2SetHi_Pass(tUnit, 0);
                tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetHi_Pass(0x%08X, 0) failed.", tUnit));
            }
            else
            {
                err = tmddTDA182I2SetHi_Pass(tUnit, 1);     
                tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetHi_Pass(0x%08X, 1) failed.", tUnit));

                if(err == TM_OK)
                {
                    /* Set IF HPF */
                    err = tmddTDA182I2SetIF_HP_Fc(tUnit, (UInt8)(pObj->Std_Array[pObj->StandardMode].IF_HPF - 1));
                    tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetIF_HP_Fc(0x%08X) failed.", tUnit));
                }
            }
        }

        if(err == TM_OK)
        {
            /* Set DC Notch */
            err = tmddTDA182I2SetIF_Notch(tUnit, pObj->Std_Array[pObj->StandardMode].DC_Notch);
            tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetIF_Notch(0x%08X) failed.", tUnit));
        }

        if(err == TM_OK)
        {
            /* Set AGC1 LNA Top */
            err = tmddTDA182I2SetAGC1_TOP(tUnit, pObj->Std_Array[pObj->StandardMode].AGC1_LNA_TOP);
            tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetAGC1_TOP(0x%08X) failed.", tUnit));
        }

        if(err == TM_OK)
        {
            /* Set AGC2 RF Top */
            err = tmddTDA182I2SetAGC2_TOP(tUnit, pObj->Std_Array[pObj->StandardMode].AGC2_RF_Attenuator_TOP);
            tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetAGC2_TOP(0x%08X) failed.", tUnit));
        }

        if(err == TM_OK)
        {
            /* Set AGC3 RF AGC Top */
            if (pObj->uRF < tmTDA182I2_AGC3_RF_AGC_TOP_FREQ_LIM)
            {
                err = tmddTDA182I2SetRFAGC_Top(tUnit, pObj->Std_Array[pObj->StandardMode].AGC3_RF_AGC_TOP_Low_band);
                tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetRFAGC_Top(0x%08X) failed.", tUnit));
            }
            else
            {
                err = tmddTDA182I2SetRFAGC_Top(tUnit, pObj->Std_Array[pObj->StandardMode].AGC3_RF_AGC_TOP_High_band);
                tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetRFAGC_Top(0x%08X) failed.", tUnit));
            }
        }

        if(err == TM_OK)
        {
            /* Set AGC4 IR Mixer Top */
            err = tmddTDA182I2SetIR_Mixer_Top(tUnit, pObj->Std_Array[pObj->StandardMode].AGC4_IR_Mixer_TOP);
            tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetIR_Mixer_Top(0x%08X) failed.", tUnit));
        }

        if(err == TM_OK)
        {
            /* Set AGC5 IF AGC Top */
            err = tmddTDA182I2SetAGC5_TOP(tUnit, pObj->Std_Array[pObj->StandardMode].AGC5_IF_AGC_TOP);
            tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetAGC5_TOP(0x%08X) failed.", tUnit));
        }

        if(err == TM_OK)
        {
            /* Set AGC3 Adapt */
            err = tmddTDA182I2SetPD_RFAGC_Adapt(tUnit, pObj->Std_Array[pObj->StandardMode].AGC3_Adapt);
            tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetPD_RFAGC_Adapt(0x%08X) failed.", tUnit));
        }
    
        if(err == TM_OK)
        {
            /* Set AGC3 Adapt TOP */
            err = tmddTDA182I2SetRFAGC_Adapt_TOP(tUnit, pObj->Std_Array[pObj->StandardMode].AGC3_Adapt_TOP);
            tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetRFAGC_Adapt_TOP(0x%08X) failed.", tUnit));
        }

        if(err == TM_OK)
        {
            /* Set AGC5 Atten 3dB */
            err = tmddTDA182I2SetRF_Atten_3dB(tUnit, pObj->Std_Array[pObj->StandardMode].AGC5_Atten_3dB);
            tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetRF_Atten_3dB(0x%08X) failed.", tUnit));
        }

        if(err == TM_OK)
        {
            /* Set AGC5 Detector HPF */
            err = tmddTDA182I2SetAGC5_Ana(tUnit, pObj->Std_Array[pObj->StandardMode].AGC5_Detector_HPF);
            tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetAGC5_Ana(0x%08X) failed.", tUnit));
        }

        if(err == TM_OK)
        {
            /* Set AGCK Mode */
            err = tmddTDA182I2SetAGCK_Mode(tUnit, pObj->Std_Array[pObj->StandardMode].GSK&0x03);
            tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetAGCK_Mode(0x%08X) failed.", tUnit));

            err = tmddTDA182I2SetAGCK_Step(tUnit, (pObj->Std_Array[pObj->StandardMode].GSK&0x0C)>>2);
            tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetAGCK_Step(0x%08X) failed.", tUnit));
        }

        if(err == TM_OK)
        {
            /* Set H3H5 VHF Filter 6 */
            err = tmddTDA182I2SetPSM_StoB(tUnit, pObj->Std_Array[pObj->StandardMode].H3H5_VHF_Filter6);
            tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetPSM_StoB(0x%08X) failed.", tUnit));
        }

        if(err == TM_OK)
        {
            /* Set IF */
            err = tmddTDA182I2SetIF_Freq(tUnit, pObj->Std_Array[pObj->StandardMode].IF - pObj->Std_Array[pObj->StandardMode].CF_Offset);
            tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetIF_Freq(0x%08X) failed.", tUnit));
        }

        /* ---------- ES2 PATCH BEGIN ---------- */
        if(err == TM_OK)
        {
            err = tmddTDA182I2GetMajorRevision(tUnit, &majorRevision);
            tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2GetMajorRevision(0x%08X) failed.", tUnit));

            err = tmddTDA182I2GetMinorRevision(tUnit, &minorRevision);
            tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2GetMinorRevision(0x%08X) failed.", tUnit));
        }

        if(err == TM_OK && majorRevision == 1 && minorRevision == 0)
        {
            /* Set LPF Gain */
            err = tmddTDA182I2SetLPF_Gain_Mode(tUnit, pObj->Std_Array[pObj->StandardMode].LPF_Gain);
            tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetLPF_Gain_Mode(0x%08X) failed.", tUnit));
        }
        /* ---------- ES2 PATCH END ---------- */

        if(err == TM_OK)
        {
            /* Set RF */
            err = tmddTDA182I2SetRF_Freq(tUnit, uRF + pObj->Std_Array[pObj->StandardMode].CF_Offset);
            tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetRF_Freq(0x%08X) failed.", tUnit));
            
            /* PATCH ES2 begin : in case of timeout WAIT IRQ */
            if(err == ddTDA182I2_ERR_NOT_READY  && bIRQWait && majorRevision == 1 && minorRevision == 0)
            {
                err = tmddTDA182I2Reset(tUnit);
                tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2Reset(0x%08X) failed.", tUnit));

                err = tmddTDA182I2SetRF_Freq(tUnit, uRF + pObj->Std_Array[pObj->StandardMode].CF_Offset);
                tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetRF_Freq(0x%08X) failed.", tUnit));
            }
            /* PATCH ES2 end */
        }
        if(err == TM_OK)
        {
            /*  patch Spurious begin */
            ratioL = (UInt8)(uRF / 16000000);
            ratioH = (UInt8)(uRF / 16000000) + 1;
            DeltaL = (uRF - (ratioL * 16000000));
            DeltaH = ((ratioH * 16000000) - uRF);

            if (uRF < 72000000 )
            {
                /* Set sigma delta clock*/
                err = tmddTDA182I2SetDigital_Clock_Mode(tUnit, 1);
                tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetDigital_Clock_Mode(0x%08X, sigma delta clock) failed.", tUnit));                    
            }
            else if (uRF < 104000000 )
            {
                /* Set 16 Mhz Xtal clock */
                err = tmddTDA182I2SetDigital_Clock_Mode(tUnit, 0);
                tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetDigital_Clock_Mode(0x%08X, 16 Mhz xtal clock) failed.", tUnit));
            }
            else if (uRF <= 120000000 )
            {
                /* Set sigma delta clock*/
                err = tmddTDA182I2SetDigital_Clock_Mode(tUnit, 1);
                tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetDigital_Clock_Mode(0x%08X, sigma delta clock) failed.", tUnit));                    
            }
            else  /* RF above 120 MHz */
            {
                if  (DeltaL <= DeltaH )  
                {
                    if (ratioL & 0x000001 )  /* ratioL odd */
                    {
                        /* Set 16 Mhz Xtal clock */
                        err = tmddTDA182I2SetDigital_Clock_Mode(tUnit, 0);
                        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetDigital_Clock_Mode(0x%08X, 16 Mhz xtal clock) failed.", tUnit));
                    }
                    else /* ratioL even */
                    {
                        /* Set sigma delta clock*/
                        err = tmddTDA182I2SetDigital_Clock_Mode(tUnit, 1);
                        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetDigital_Clock_Mode(0x%08X, sigma delta clock) failed.", tUnit));
                    }
                    
                }
                else  /* (DeltaL > DeltaH ) */
                {
                    if (ratioL & 0x000001 )  /*(ratioL odd)*/
                    {
                        /* Set sigma delta clock*/
                        err = tmddTDA182I2SetDigital_Clock_Mode(tUnit, 1);
                        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetDigital_Clock_Mode(0x%08X, sigma delta clock) failed.", tUnit));
                    }
                    else
                    {
                        /* Set 16 Mhz Xtal clock */
                        err = tmddTDA182I2SetDigital_Clock_Mode(tUnit, 0);
                        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetDigital_Clock_Mode(0x%08X, 16 Mhz xtal clock) failed.", tUnit));
                    }
                }
            }
        }
        /*  patch Spurious end */
        (void)TDA182I2MutexRelease(pObj);
    }

    return err;
}


//-------------------------------------------------------------------------------------
// FUNCTION:    tmbslTDA182I2GetRf:
//
// DESCRIPTION: Get the frequency programmed in the tuner
//
// RETURN:      TMBSL_ERR_TUNER_BAD_UNIT_NUMBER
//              TDA182I2_ERR_NOT_INITIALIZED
//              TM_OK
//
// NOTES:       The value returned is the one stored in the Object
//-------------------------------------------------------------------------------------
//
tmErrorCode_t
tmbslTDA182I2GetRf
(
    tmUnitSelect_t  tUnit,  /* I: Unit number */
    UInt32*         puRF    /* O: RF frequency in hertz */
)
{
    ptmTDA182I2Object_t pObj = Null;
    tmErrorCode_t       err  = TM_OK;

    if(puRF == Null)
        err = TDA182I2_ERR_BAD_PARAMETER;

    //------------------------------
    // test input parameters
    //------------------------------
    // pObj initialization
    if(err == TM_OK)
    {
        err = TDA182I2GetInstance(tUnit, &pObj);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "TDA182I2GetInstance(0x%08X) failed.", tUnit));
    }

    if(err == TM_OK)
    {
        err = TDA182I2MutexAcquire(pObj, TDA182I2_MUTEX_TIMEOUT);
    }

    if(err == TM_OK)
    {
        /* Get RF */
        *puRF = pObj->uRF/* - pObj->Std_Array[pObj->StandardMode].CF_Offset*/;

        (void)TDA182I2MutexRelease(pObj);
    }
    return err;
}

/*============================================================================*/
/* tmbslTDA182I2Reset                                                         */
/*============================================================================*/
tmErrorCode_t
tmbslTDA182I2Reset
(
    tmUnitSelect_t  tUnit   /* I: Unit number */
)
{
    ptmTDA182I2Object_t pObj = Null;
    tmErrorCode_t       err = TM_OK;
    Bool                bIRQWait = True;
    /* ---------- ES2 PATCH BEGIN ---------- */
    UInt8               majorRevision = 0;
    UInt8               minorRevision = 0;
    /* ---------- ES2 PATCH END ---------- */

    //------------------------------
    // test input parameters
    //------------------------------
    // pObj initialization
    err = TDA182I2GetInstance(tUnit, &pObj);
    tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "TDA182I2GetInstance(0x%08X) failed.", tUnit));

    if(err == TM_OK)
    {
        err = TDA182I2MutexAcquire(pObj, TDA182I2_MUTEX_TIMEOUT);
    }

    if(err == TM_OK)
    {
        err = tmddTDA182I2GetIRQWait(tUnit, &bIRQWait);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2GetIRQWait(0x%08X) failed.", tUnit));
    }

    if(err == TM_OK)
    {
        err = TDA182I2Init(tUnit);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "TDA182I2Init(0x%08X) failed.", tUnit));
    }

    if(err == TM_OK)
    {
        err = tmddTDA182I2GetMajorRevision(tUnit, &majorRevision);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2GetMajorRevision(0x%08X) failed.", tUnit));

        err = tmddTDA182I2GetMinorRevision(tUnit, &minorRevision);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2GetMinorRevision(0x%08X) failed.", tUnit));
    }

    if(err == TM_OK)
    {
        // initialize tuner
        err =  tmddTDA182I2Reset(tUnit);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2Reset(0x%08X) failed.", tUnit));


        /* PATCH ES2 begin : in case of timeout WAIT IRQ */
        if(err == ddTDA182I2_ERR_NOT_READY && bIRQWait && majorRevision == 1 && minorRevision == 0)
        {
            err = tmddTDA182I2Reset(tUnit);
            tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2Reset(0x%08X) failed.", tUnit));
        }
        /* PATCH ES2 end */

        /****** Begin settings V3.0.1 *********/
        
#ifdef TMBSL_TDA18272
        /* LT_Enable  N     0*/
        if (err == TM_OK)
        {
            err = tmddTDA182I2SetLT_Enable(tUnit, 0 );
            tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetLT_Enable(0x%08X, 0) failed.", tUnit));
        }
        /* PSM AGC1   -5,9 mA     1*/
        if (err == TM_OK)
        {
            err = tmddTDA182I2SetPSM_AGC1(tUnit, 1 );
            tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetPSM_AGC1(0x%08X, 1) failed.", tUnit));
        }

        /* AGC1 gain range  [-12, +15dB]      0*/
        if (err == TM_OK)
        {
            err = tmddTDA182I2SetAGC1_6_15dB(tUnit, 0 );
            tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetAGC1_6_15dB(0x%08X, 0) failed.", tUnit));
        }
        /****** End settings V3.0.1 *********/
#endif


    }
    (void)TDA182I2MutexRelease(pObj);

    return err;
}

/*============================================================================*/
/* tmbslTDA182I2GetIF                                                         */
/*============================================================================*/

tmErrorCode_t
tmbslTDA182I2GetIF
(
    tmUnitSelect_t  tUnit,  /* I: Unit number */
    UInt32*         puIF    /* O: IF Frequency in hertz */
)
{
    ptmTDA182I2Object_t pObj = Null;
    tmErrorCode_t       err = TM_OK;

    if(puIF == Null)
        err = TDA182I2_ERR_BAD_PARAMETER;
    
    if(err == TM_OK)
    {
        err = TDA182I2GetInstance(tUnit, &pObj);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "TDA182I2GetInstance(0x%08X) failed.", tUnit));
    }

    if(err == TM_OK)
    {
        err = TDA182I2MutexAcquire(pObj, TDA182I2_MUTEX_TIMEOUT);
    }

    if(err == TM_OK)
    {
        *puIF = pObj->Std_Array[pObj->StandardMode].IF - pObj->Std_Array[pObj->StandardMode].CF_Offset;

        (void)TDA182I2MutexRelease(pObj);
    }

    return err;
}

/*============================================================================*/
/* tmbslTDA182I2GetCF_Offset                                                  */
/*============================================================================*/

tmErrorCode_t
tmbslTDA182I2GetCF_Offset(
    tmUnitSelect_t  tUnit,      /* I: Unit number */
    UInt32*         puOffset    /* O: Center frequency offset in hertz */
    )
{
    ptmTDA182I2Object_t pObj = Null;
    tmErrorCode_t       err = TM_OK;

    if(puOffset == Null)
        err = TDA182I2_ERR_BAD_PARAMETER;
    
    if(err == TM_OK)
    {
        err = TDA182I2GetInstance(tUnit, &pObj);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "TDA182I2GetInstance(0x%08X) failed.", tUnit));
    }

    if(err == TM_OK)
    {
        err = TDA182I2MutexAcquire(pObj, TDA182I2_MUTEX_TIMEOUT);
    }

    if(err == TM_OK)
    {
        *puOffset = pObj->Std_Array[pObj->StandardMode].CF_Offset;

        (void)TDA182I2MutexRelease(pObj);
    }

    return err;
}

/*============================================================================*/
/* tmbslTDA182I2GetLockStatus                                                 */
/*============================================================================*/

tmErrorCode_t
tmbslTDA182I2GetLockStatus
(
    tmUnitSelect_t          tUnit,      /* I: Unit number */
    tmbslFrontEndState_t*   pLockStatus /* O: PLL Lock status */
)
{
    ptmTDA182I2Object_t pObj = Null;
    tmErrorCode_t       err = TM_OK;
    UInt8 uValue, uValueLO;

    if(pLockStatus == Null)
        err = TDA182I2_ERR_BAD_PARAMETER;

    if(err == TM_OK)
    {
        err = TDA182I2GetInstance(tUnit, &pObj);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "TDA182I2GetInstance(0x%08X) failed.", tUnit));
    }

    if(err == TM_OK)
    {
        err = TDA182I2MutexAcquire(pObj, TDA182I2_MUTEX_TIMEOUT);
    }
    if(err == TM_OK)
    {
        err =  tmddTDA182I2GetLO_Lock(tUnit, &uValueLO);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2GetLO_Lock(0x%08X) failed.", tUnit));
    }
    if(err == TM_OK)
    {
        err =  tmddTDA182I2GetIRQ_status(tUnit, &uValue);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2GetIRQ_status(0x%08X) failed.", tUnit));

        uValue = uValue & uValueLO;
    }
    if(err == TM_OK)
    {
        *pLockStatus =  (uValue)? tmbslFrontEndStateLocked : tmbslFrontEndStateNotLocked;
    }
    else
    {
        *pLockStatus = tmbslFrontEndStateUnknown;  
    }

    (void)TDA182I2MutexRelease(pObj);

    return err;
}

/*============================================================================*/
/* tmbslTDA182I2GetPowerLevel                                                 */
/*============================================================================*/

tmErrorCode_t
tmbslTDA182I2GetPowerLevel
(
    tmUnitSelect_t  tUnit,      /* I: Unit number */
    UInt32*         pPowerLevel /* O: Power Level in dB�V */
)
{
    ptmTDA182I2Object_t pObj = Null;
    tmErrorCode_t       err = TM_OK;

    if(pPowerLevel == Null)
        err = TDA182I2_ERR_BAD_PARAMETER;

    if(err == TM_OK)
    {
        err = TDA182I2GetInstance(tUnit, &pObj);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "TDA182I2GetInstance(0x%08X) failed.", tUnit));
    }

    if(err == TM_OK)
    {
        err = TDA182I2MutexAcquire(pObj, TDA182I2_MUTEX_TIMEOUT);
    }

    if(err == TM_OK)
    {
        *pPowerLevel = 0;

        err = tmddTDA182I2GetPower_Level(tUnit, (UInt8 *)pPowerLevel);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2GetPower_Level(0x%08X) failed.", tUnit));

        (void)TDA182I2MutexRelease(pObj);
    }
    return err;
}

/*============================================================================*/
/* tmbslTDA182I2SetIRQWait                                                  */
/*============================================================================*/

tmErrorCode_t
tmbslTDA182I2SetIRQWait
(
    tmUnitSelect_t  tUnit,  /* I: Unit number */
    Bool            bWait   /* I: Determine if we need to wait IRQ in driver functions */
)
{
    ptmTDA182I2Object_t pObj = Null;
    tmErrorCode_t       err = TM_OK;

    err = TDA182I2GetInstance(tUnit, &pObj);
    tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "TDA182I2GetInstance(0x%08X) failed.", tUnit));

    if(err == TM_OK)
    {
        err = TDA182I2MutexAcquire(pObj, TDA182I2_MUTEX_TIMEOUT);
    }

    if(err == TM_OK)
    {
        err = tmddTDA182I2SetIRQWait(tUnit, bWait);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SetIRQWait(0x%08X) failed.", tUnit));

        (void)TDA182I2MutexRelease(pObj);
    }
    return err;
}

/*============================================================================*/
/* tmbslTDA182I2GetIRQWait                                                  */
/*============================================================================*/

tmErrorCode_t
tmbslTDA182I2GetIRQWait
(
    tmUnitSelect_t  tUnit,  /* I: Unit number */
    Bool*           pbWait  /* O: Determine if we need to wait IRQ in driver functions */
)
{
    ptmTDA182I2Object_t pObj = Null;
    tmErrorCode_t       err = TM_OK;

    if(pbWait == Null)
        err = TDA182I2_ERR_BAD_PARAMETER;

    if(err == TM_OK)
    {
        err = TDA182I2GetInstance(tUnit, &pObj);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "TDA182I2GetInstance(0x%08X) failed.", tUnit));
    }

    if(err == TM_OK)
    {
        err = TDA182I2MutexAcquire(pObj, TDA182I2_MUTEX_TIMEOUT);
    }

    if(err == TM_OK)
    {
        err = tmddTDA182I2GetIRQWait(tUnit, pbWait);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2GetIRQWait(0x%08X) failed.", tUnit));

        (void)TDA182I2MutexRelease(pObj);
    }

    return err;
}

/*============================================================================*/
/* tmbslTDA182I2GetIRQ                                                        */
/*============================================================================*/

tmErrorCode_t
tmbslTDA182I2GetIRQ
(
    tmUnitSelect_t  tUnit  /* I: Unit number */,
    Bool*           pbIRQ  /* O: IRQ triggered */
)
{
    ptmTDA182I2Object_t pObj = Null;
    tmErrorCode_t       err = TM_OK;

    if(pbIRQ == Null)
        err = TDA182I2_ERR_BAD_PARAMETER;

    if(err == TM_OK)
    {
        err = TDA182I2GetInstance(tUnit, &pObj);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "TDA182I2GetInstance(0x%08X) failed.", tUnit));
    }

    if(err == TM_OK)
    {
        err = TDA182I2MutexAcquire(pObj, TDA182I2_MUTEX_TIMEOUT);
    }

    if(err == TM_OK)
    {
        *pbIRQ = 0;

        err = tmddTDA182I2GetIRQ_status(tUnit, (UInt8 *)pbIRQ);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2GetIRQ_status(0x%08X) failed.", tUnit));

        (void)TDA182I2MutexRelease(pObj);
    }

    return err;
}

/*============================================================================*/
/* tmbslTDA182I2WaitIRQ                                                       */
/*============================================================================*/

tmErrorCode_t
tmbslTDA182I2WaitIRQ
(
    tmUnitSelect_t  tUnit,      /* I: Unit number */
    UInt32          timeOut,    /* I: timeOut for IRQ wait */
    UInt32          waitStep,   /* I: wait step */
    UInt8           irqStatus   /* I: IRQs to wait */
)
{
    ptmTDA182I2Object_t pObj = Null;
    tmErrorCode_t       err = TM_OK;

    if(err == TM_OK)
    {
        err = TDA182I2GetInstance(tUnit, &pObj);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "TDA182I2GetInstance(0x%08X) failed.", tUnit));
    }

    if(err == TM_OK)
    {
        err = TDA182I2MutexAcquire(pObj, TDA182I2_MUTEX_TIMEOUT);
    }

    if(err == TM_OK)
    {
        err = tmddTDA182I2WaitIRQ(tUnit, timeOut, waitStep, irqStatus);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2WaitIRQ(0x%08X) failed.", tUnit));

        (void)TDA182I2MutexRelease(pObj);
    }

    return err;
}

/*============================================================================*/
/* tmbslTDA182I2GetXtalCal_End                                                */
/*============================================================================*/

tmErrorCode_t
tmbslTDA182I2GetXtalCal_End
(
    tmUnitSelect_t  tUnit  /* I: Unit number */,
    Bool*           pbXtalCal_End  /* O: XtalCal_End triggered */
)
{
    ptmTDA182I2Object_t pObj = Null;
    tmErrorCode_t       err = TM_OK;

    if(pbXtalCal_End == Null)
        err = TDA182I2_ERR_BAD_PARAMETER;

    if(err == TM_OK)
    {
        err = TDA182I2GetInstance(tUnit, &pObj);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "TDA182I2GetInstance(0x%08X) failed.", tUnit));
    }

    if(err == TM_OK)
    {
        err = TDA182I2MutexAcquire(pObj, TDA182I2_MUTEX_TIMEOUT);
    }

    if(err == TM_OK)
    {
        *pbXtalCal_End = 0;

        err = tmddTDA182I2GetMSM_XtalCal_End(tUnit, (UInt8 *)pbXtalCal_End);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2GetMSM_XtalCal_End(0x%08X) failed.", tUnit));

        (void)TDA182I2MutexRelease(pObj);
    }

    return err;
}

/*============================================================================*/
/* tmbslTDA182I2WaitXtalCal_End                                               */
/*============================================================================*/

tmErrorCode_t
tmbslTDA182I2WaitXtalCal_End
(
    tmUnitSelect_t  tUnit,      /* I: Unit number */
    UInt32          timeOut,    /* I: timeOut for IRQ wait */
    UInt32          waitStep   /* I: wait step */
)
{
    ptmTDA182I2Object_t pObj = Null;
    tmErrorCode_t       err = TM_OK;

    if(err == TM_OK)
    {
        err = TDA182I2GetInstance(tUnit, &pObj);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "TDA182I2GetInstance(0x%08X) failed.", tUnit));
    }

    if(err == TM_OK)
    {
        err = TDA182I2MutexAcquire(pObj, TDA182I2_MUTEX_TIMEOUT);
    }

    if(err == TM_OK)
    {
        err = tmddTDA182I2WaitXtalCal_End(tUnit, timeOut, waitStep);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2WaitXtalCal_End(0x%08X) failed.", tUnit));

        (void)TDA182I2MutexRelease(pObj);
    }

    return err;
}

/*============================================================================*/
/* tmbslTDA182I2SoftReset                                                     */
/*============================================================================*/

tmErrorCode_t
tmbslTDA182I2SoftReset
(
    tmUnitSelect_t  tUnit   /* I: Unit number */
)
{
    ptmTDA182I2Object_t pObj = Null;
    tmErrorCode_t       err = TM_OK;

    if(err == TM_OK)
    {
        err = TDA182I2GetInstance(tUnit, &pObj);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "TDA182I2GetInstance(0x%08X) failed.", tUnit));
    }

    if(err == TM_OK)
    {
        err = TDA182I2MutexAcquire(pObj, TDA182I2_MUTEX_TIMEOUT);
    }

    if(err == TM_OK)
    {
        err = tmddTDA182I2SoftReset(tUnit);
        tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "tmddTDA182I2SoftReset(0x%08X) failed.", tUnit));

        (void)TDA182I2MutexRelease(pObj);
    }

    return err;
}



/*============================================================================*/
/* tmbslTDA182I2Write                                                         */
/*============================================================================*/

tmErrorCode_t
tmbslTDA182I2Write
(
    tmUnitSelect_t  tUnit,      /* I: Unit number */
    UInt32          uIndex,     /* I: Start index to write */
    UInt32          WriteLen,   /* I: Number of bytes to write */
    UInt8*          pData       /* I: Data to write */
)
{
    ptmTDA182I2Object_t pObj = Null;
    tmErrorCode_t       err  = TM_OK;

    // pObj initialization
    err = TDA182I2GetInstance(tUnit, &pObj);
    tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "TDA182I2GetInstance(0x%08X) failed.", tUnit));

    if(err == TM_OK)
    {
        err = TDA182I2MutexAcquire(pObj, TDA182I2_MUTEX_TIMEOUT);
    }

    if(err == TM_OK)
    {
        // Call tmddTDA182I2Write

        (void)TDA182I2MutexRelease(pObj);
    }

    return err;
}

/*============================================================================*/
/* tmbslTDA182I2Read                                                          */
/*============================================================================*/

tmErrorCode_t
tmbslTDA182I2Read
(
    tmUnitSelect_t  tUnit,      /* I: Unit number */
    UInt32          uIndex,     /* I: Start index to read */
    UInt32          ReadLen,    /* I: Number of bytes to read */
    UInt8*          pData       /* I: Data to read */
)
{
    ptmTDA182I2Object_t pObj = Null;
    tmErrorCode_t       err  = TM_OK;

    // pObj initialization
    err = TDA182I2GetInstance(tUnit, &pObj);
    tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "TDA182I2GetInstance(0x%08X) failed.", tUnit));

    if(err == TM_OK)
    {
        err = TDA182I2MutexAcquire(pObj, TDA182I2_MUTEX_TIMEOUT);
    }

    if(err == TM_OK)
    {
        // Call tmddTDA182I2Read

        (void)TDA182I2MutexRelease(pObj);
    }

    return err;
}

//-----------------------------------------------------------------------------
// Internal functions:
//-----------------------------------------------------------------------------
//

//-----------------------------------------------------------------------------
// FUNCTION:    TDA182I2Init:
//
// DESCRIPTION: initialization of the Tuner
//
// RETURN:      always True
//
// NOTES:       
//-----------------------------------------------------------------------------
//
static tmErrorCode_t
TDA182I2Init
(
    tmUnitSelect_t  tUnit   /* I: Unit number */
)
{     
    ptmTDA182I2Object_t pObj = Null;
    tmErrorCode_t       err  = TM_OK;

    //------------------------------
    // test input parameters
    //------------------------------
    // pObj initialization
    err = TDA182I2GetInstance(tUnit, &pObj);
    tmASSERTExT(err, TM_OK, (DEBUGLVL_ERROR, "TDA182I2GetInstance(0x%08X) failed.", tUnit));

    if(err == TM_OK)
    {
        //err = tmddTDA182I2SetIRQWait(tUnit, True);

        //if(pObj->bIRQWait)
        //{
        //    err = TDA182I2WaitIRQ(pObj);
        //}
    }

    return err;
}

//-------------------------------------------------------------------------------------
// FUNCTION:    TDA182I2Wait
//
// DESCRIPTION: This function waits for requested time
//
// RETURN:      True or False
//
// NOTES:       
//-------------------------------------------------------------------------------------
//
static tmErrorCode_t 
TDA182I2Wait
(
    ptmTDA182I2Object_t pObj,   /* I: Driver object */
    UInt32              Time   /*  I: Time to wait for */
)
{
    tmErrorCode_t   err  = TM_OK;

    // wait Time ms
    err = POBJ_SRVFUNC_STIME.Wait(pObj->tUnit, Time);

    // Return value
    return err;
}

/*============================================================================*/
/* TDA182I2MutexAcquire                                                       */
/*============================================================================*/
extern tmErrorCode_t
TDA182I2MutexAcquire
(
 ptmTDA182I2Object_t    pObj,
 UInt32                 timeOut
 )
{
    tmErrorCode_t       err = TM_OK;

    if(pObj->sMutex.Acquire != Null && pObj->pMutex != Null)
    {
        err = pObj->sMutex.Acquire(pObj->pMutex, timeOut);
    }

    return err;
}

/*============================================================================*/
/* TDA182I2MutexRelease                                                       */
/*============================================================================*/
extern tmErrorCode_t
TDA182I2MutexRelease
(
 ptmTDA182I2Object_t    pObj
 )
{
    tmErrorCode_t       err = TM_OK;

    if(pObj)
    {
        if(pObj->sMutex.Release != Null && pObj->pMutex != Null)
        {
            err = pObj->sMutex.Release(pObj->pMutex);
        }
    }
    

    return err;
}

