/**

@file

@brief   RTD2885 ATSC demod module definition

One can manipulate RTD2885 ATSC demod through RTD2885 ATSC module.
RTD2885 ATSC module is derived from ATSC demod module.

*/


#include "demod_rtd2885_atsc.h"





/**

@brief   RTD2885 ATSC demod module builder

Use BuildRtd2885AtscModule() to build RTD2885 ATSC module, set all module function pointers with the corresponding
functions, and initialize module private variables.


@param [in]   ppDemod                      Pointer to RTD2885 ATSC demod module pointer
@param [in]   pAtscDemodModuleMemory       Pointer to an allocated ATSC demod module memory
@param [in]   pBaseInterfaceModuleMemory   Pointer to an allocated base interface module memory
@param [in]   pI2cBridgeModuleMemory       Pointer to an allocated I2C bridge module memory
@param [in]   DeviceAddr                   RTD2885 ATSC I2C device address
@param [in]   CrystalFreqHz                RTD2885 ATSC crystal frequency in Hz
@param [in]   TsInterfaceMode              RTD2885 TS interface mode for setting


@note
    -# One should call BuildRtd2885AtscModule() to build RTD2885 ATSC module before using it.

*/
void
BuildRtd2885AtscModule(
    ATSC_DEMOD_MODULE **ppDemod,
    ATSC_DEMOD_MODULE *pAtscDemodModuleMemory,
    BASE_INTERFACE_MODULE *pBaseInterfaceModuleMemory,
    I2C_BRIDGE_MODULE *pI2cBridgeModuleMemory,
    unsigned char DeviceAddr,
    unsigned long CrystalFreqHz,
    int TsInterfaceMode
    )
{
    ATSC_DEMOD_MODULE *pDemod;



    // Set demod module pointer.
    *ppDemod = pAtscDemodModuleMemory;

    // Get demod module.
    pDemod = *ppDemod;

    // Set base interface module pointer and I2C bridge module pointer.
    pDemod->pExtra         = INVALID_POINTER_VALUE;
    pDemod->pBaseInterface = pBaseInterfaceModuleMemory;
    pDemod->pI2cBridge     = pI2cBridgeModuleMemory;


    // Set demod type.
    pDemod->DemodType = ATSC_DEMOD_TYPE_RTD2885_ATSC;

    // Set demod I2C device address.
    pDemod->DeviceAddr = DeviceAddr;

    // Set demod crystal frequency in Hz.
    pDemod->CrystalFreqHz = CrystalFreqHz;

    // Set demod TS interface mode.
    pDemod->TsInterfaceMode = TsInterfaceMode;


    // Initialize demod parameter setting status
    pDemod->IsSymbolRateHzSet  = NO;
    pDemod->IsIfFreqHzSet      = NO;
    pDemod->IsSpectrumModeSet  = NO;


    // Initialize register tables in demod module.
    rtd2885_atsc_InitRegTable(pDemod);


    // Build I2C birdge module.
    rtd2885_atsc_BuildI2cBridgeModule(pDemod);


    // Set demod module I2C function pointers with default functions.
    pDemod->SetRegPage         = atsc_demod_default_SetRegPage;
    pDemod->SetRegBytes        = atsc_demod_default_SetRegBytes;
    pDemod->GetRegBytes        = atsc_demod_default_GetRegBytes;
    pDemod->SetRegMaskBits     = atsc_demod_default_SetRegMaskBits;
    pDemod->GetRegMaskBits     = atsc_demod_default_GetRegMaskBits;
    pDemod->SetRegBits         = atsc_demod_default_SetRegBits;
    pDemod->GetRegBits         = atsc_demod_default_GetRegBits;
    pDemod->SetRegBitsWithPage = atsc_demod_default_SetRegBitsWithPage;
    pDemod->GetRegBitsWithPage = atsc_demod_default_GetRegBitsWithPage;


    // Set demod module manipulating function pointers with default functions.
    pDemod->GetDemodType     = atsc_demod_default_GetDemodType;
    pDemod->GetDeviceAddr    = atsc_demod_default_GetDeviceAddr;
    pDemod->GetCrystalFreqHz = atsc_demod_default_GetCrystalFreqHz;

    pDemod->GetSymbolRateHz  = atsc_demod_default_GetSymbolRateHz;
    pDemod->GetIfFreqHz      = atsc_demod_default_GetIfFreqHz;
    pDemod->GetSpectrumMode  = atsc_demod_default_GetSpectrumMode;


    // Set demod module manipulating function pointers with particular functions.
    pDemod->IsConnectedToI2c  = rtd2885_atsc_IsConnectedToI2c;
    pDemod->SoftwareReset     = rtd2885_atsc_SoftwareReset;

    pDemod->Initialize        = rtd2885_atsc_Initialize;
    pDemod->SetSymbolRateHz   = rtd2885_atsc_SetSymbolRateHz;
    pDemod->SetIfFreqHz       = rtd2885_atsc_SetIfFreqHz;
    pDemod->SetSpectrumMode   = rtd2885_atsc_SetSpectrumMode;

    pDemod->GetRfAgc          = rtd2885_atsc_GetRfAgc;
    pDemod->GetIfAgc          = rtd2885_atsc_GetIfAgc;
    pDemod->GetDiAgc          = rtd2885_atsc_GetDiAgc;
    pDemod->GetTrOffsetPpm    = rtd2885_atsc_GetTrOffsetPpm;
    pDemod->GetCrOffsetHz     = rtd2885_atsc_GetCrOffsetHz;

    pDemod->IsAagcLocked      = rtd2885_atsc_IsAagcLocked;
    pDemod->IsFrameLocked     = rtd2885_atsc_IsFrameLocked;

    pDemod->GetErrorRate      = rtd2885_atsc_GetErrorRate;
    pDemod->GetSnrDb          = rtd2885_atsc_GetSnrDb;

    pDemod->GetSignalStrength = rtd2885_atsc_GetSignalStrength;
    pDemod->GetSignalQuality  = rtd2885_atsc_GetSignalQuality;

    pDemod->UpdateFunction    = rtd2885_atsc_UpdateFunction;
    pDemod->ResetFunction     = rtd2885_atsc_ResetFunction;


    return;
}





/**

@see   ATSC_DEMOD_FP_IS_CONNECTED_TO_I2C

*/
void
rtd2885_atsc_IsConnectedToI2c(
    ATSC_DEMOD_MODULE *pDemod,
    int *pAnswer
    )
{
    unsigned long ReadingValue;



    // Set reading value to zero, and get SYS_VERSION value.
    // Note: Use GetRegBitsWithPage() to get register bits with page setting.
    ReadingValue = 0;

    if(pDemod->GetRegBitsWithPage(pDemod, ATSC_SYS_VERSION, &ReadingValue) != FUNCTION_SUCCESS)
        goto error_status_get_demod_registers;


    // Compare SYS_VERSION value with RTD2885_ATSC_SYS_VERSION_VALUE.
    if(ReadingValue == RTD2885_ATSC_SYS_VERSION_VALUE)
        *pAnswer = YES;
    else
        *pAnswer = NO;


    return;


error_status_get_demod_registers:

    *pAnswer = NO;

    return;
}





/**

@see   ATSC_DEMOD_FP_SOFTWARE_RESET

*/
int
rtd2885_atsc_SoftwareReset(
    ATSC_DEMOD_MODULE *pDemod
    )
{
    // Set register page number with system page number for software resetting.
    if(pDemod->SetRegPage(pDemod, 0) != FUNCTION_SUCCESS)
        goto error_status_set_demod_registers;


    // Set and clear SOFT_RESET register bit.
    if(pDemod->SetRegBits(pDemod, ATSC_OPT_SOFT_RESET_ATSC, ON) != FUNCTION_SUCCESS)
        goto error_status_set_demod_registers;

    if(pDemod->SetRegBits(pDemod, ATSC_OPT_SOFT_RESET_ATSC, OFF) != FUNCTION_SUCCESS)
        goto error_status_set_demod_registers;


    return FUNCTION_SUCCESS;


error_status_set_demod_registers:
    return FUNCTION_ERROR;
}





/**

@see   ATSC_DEMOD_FP_INITIALIZE

*/
int
rtd2885_atsc_Initialize(
    ATSC_DEMOD_MODULE *pDemod
    )
{
    typedef struct
    {
        unsigned char PageNo;
        unsigned char RegStartAddr;
        unsigned char Msb;
        unsigned char Lsb;
        unsigned long WritingValue;
    }
    INIT_REG_ENTRY;


    typedef struct
    {
        int RegBitName;
        unsigned long WritingValue[TS_INTERFACE_MODE_NUM];
    }
    TS_INTERFACE_INIT_TABLE_ENTRY;



    static const INIT_REG_ENTRY InitRegTable[RTD2885_ATSC_INIT_REG_TABLE_LEN] =
    {
        // PageNo,  RegStartAddr,   Msb,    Lsb,    WritingValue
        {0x0,       0x6,        6,      6,      0x0         },
        {0x28,      0x53,       0,      0,      0x1         },
        {0x21,      0x53,       8,      0,      0x1c8       },
        {0x21,      0xc7,       0,      0,      0x1         },
        {0x21,      0xc7,       1,      1,      0x1         },
        {0x21,      0xc7,       2,      2,      0x1         },
        {0x26,      0x38,       3,      0,      0x8         },
        {0x26,      0x4a,       2,      0,      0x0         },
        {0x26,      0xb0,       12,     0,      0x258       },
        {0x27,      0x15,       31,     0,      0xfffc0000  },
        {0x27,      0x19,       7,      0,      0x6         },
        {0x27,      0x1a,       11,     0,      0x33        },
        {0x27,      0x1c,       0,      0,      0x0         },
        {0x27,      0x1c,       13,     2,      0x33        },
        {0x27,      0x30,       2,      2,      0x0         },
        {0x27,      0x31,       3,      0,      0x1         },
        {0x27,      0x31,       7,      4,      0x1         },
        {0x27,      0x50,       2,      2,      0x0         },
        {0x27,      0x51,       3,      0,      0x1         },
        {0x27,      0x51,       7,      4,      0x1         },
        {0x27,      0x56,       11,     0,      0x333       },
        {0x27,      0x57,       15,     4,      0x19a       },
        {0x27,      0x59,       11,     0,      0xcd        },
        {0x27,      0x5a,       15,     4,      0x66        },
        {0x27,      0x5c,       11,     0,      0x33        },
        {0x27,      0x79,       7,      0,      0xd         },
        {0x27,      0x7a,       11,     0,      0x66        },
        {0x27,      0x7c,       0,      0,      0x0         },
        {0x27,      0x7c,       13,     2,      0x66        },
        {0x27,      0xa1,       15,     0,      0x1400      },
        {0x28,      0x2,        7,      0,      0xf0        },
        {0x28,      0x14,       20,     0,      0x108000    },
        {0x28,      0x16,       5,      5,      0x0         },
        {0x28,      0x20,       9,      0,      0x19        },
        {0x28,      0x22,       9,      0,      0x1c        },
        {0x28,      0x24,       5,      0,      0x3         },
        {0x28,      0x26,       11,     2,      0x19        },
        {0x28,      0x28,       9,      0,      0x1c        },
        {0x28,      0x2c,       9,      0,      0x19        },
        {0x28,      0x2e,       9,      0,      0x1c        },
        {0x21,      0x4,        5,      5,      0x0         },
        {0x21,      0x7,        7,      0,      0x7f        },
        {0x21,      0xa,        7,      0,      0x80        },
        {0x21,      0xb,        7,      0,      0x7f        },
        {0x21,      0xc,        7,      0,      0x0         },
        {0x21,      0xd,        7,      0,      0x0         },
        {0x21,      0x11,       4,      4,      0x1         },
        {0x21,      0x12,       9,      0,      0x3f9       },
        {0x21,      0x14,       9,      0,      0x3ed       },
        {0x21,      0x16,       9,      0,      0x3e8       },
        {0x21,      0x18,       9,      0,      0x3f6       },
        {0x21,      0x1a,       9,      0,      0x25        },
        {0x21,      0x1c,       9,      0,      0x6e        },
        {0x21,      0x1e,       9,      0,      0xbb        },
        {0x21,      0x20,       9,      0,      0xec        },
        {0x21,      0x46,       7,      6,      0x1         },
        {0x21,      0x47,       0,      0,      0x0         },
        {0x21,      0x65,       26,     0,      0x2823eff   },
        {0x21,      0x6e,       7,      0,      0x12        },
        {0x21,      0x6f,       7,      0,      0x10        },
        {0x21,      0x70,       7,      0,      0x1         },
        {0x21,      0x71,       7,      0,      0x2         },
        {0x21,      0x73,       2,      2,      0x1         },
        {0x21,      0x73,       4,      3,      0x1         },
        {0x21,      0x73,       7,      5,      0x2         },
        {0x21,      0x75,       9,      0,      0x1a        },
        {0x21,      0x76,       4,      2,      0x0         },
        {0x21,      0x76,       5,      5,      0x0         },
        {0x21,      0x78,       3,      0,      0x2         },
        {0x21,      0x78,       4,      4,      0x1         },
        {0x21,      0x90,       11,     0,      0x3         },
        {0x21,      0x92,       11,     0,      0x7         },
        {0x21,      0x94,       0,      0,      0x1         },
        {0x21,      0xa4,       15,     0,      0xa         },
        {0x21,      0xa6,       15,     0,      0xa         },
        {0x21,      0xa8,       5,      0,      0xb         },
        {0x21,      0xa9,       5,      0,      0x6         },
        {0x21,      0xaa,       7,      0,      0x4         },
        {0x21,      0xab,       5,      0,      0x26        },
        {0x21,      0xac,       5,      0,      0x14        },
        {0x21,      0xad,       7,      0,      0x6         },
        {0x21,      0xaf,       4,      4,      0x1         },
        {0x21,      0xaf,       5,      5,      0x1         },
        {0x21,      0xb4,       3,      3,      0x1         },
        {0x21,      0xc7,       0,      0,      0x0         },
        {0x21,      0xc7,       1,      1,      0x0         },
        {0x21,      0xc7,       2,      2,      0x0         },
        {0x22,      0x1,        3,      3,      0x1         },
        {0x22,      0x1,        6,      4,      0x4         },
        {0x22,      0x20,       1,      0,      0x0         },
        {0x22,      0x20,       3,      2,      0x0         },
        {0x22,      0x21,       5,      0,      0x4         },
        {0x25,      0x1,        10,     0,      0x7fc       },
        {0x25,      0x2,        7,      3,      0x1         },
        {0x25,      0x4,        7,      3,      0x3         },
        {0x25,      0x5,        10,     0,      0x16        },
        {0x25,      0x6,        7,      3,      0x4         },
        {0x25,      0x7,        10,     0,      0x17        },
        {0x25,      0x8,        7,      3,      0x2         },
        {0x25,      0x9,        10,     0,      0xb         },
        {0x25,      0xa,        7,      3,      0x0         },
        {0x25,      0xb,        10,     0,      0x7f7       },
        {0x25,      0xc,        7,      3,      0x1d        },
        {0x25,      0xd,        10,     0,      0x7e5       },
        {0x25,      0xe,        7,      3,      0x1c        },
        {0x25,      0xf,        10,     0,      0x7de       },
        {0x25,      0x10,       7,      3,      0x1d        },
        {0x25,      0x11,       10,     0,      0x7e8       },
        {0x25,      0x12,       7,      3,      0x0         },
        {0x25,      0x13,       10,     0,      0x1         },
        {0x25,      0x14,       7,      3,      0x3         },
        {0x25,      0x15,       10,     0,      0x1e        },
        {0x25,      0x16,       7,      3,      0x4         },
        {0x25,      0x17,       10,     0,      0x30        },
        {0x25,      0x18,       7,      3,      0x3         },
        {0x25,      0x19,       10,     0,      0x2b        },
        {0x25,      0x1a,       7,      3,      0x0         },
        {0x25,      0x1b,       10,     0,      0xe         },
        {0x25,      0x1c,       7,      3,      0x1d        },
        {0x25,      0x1d,       10,     0,      0x7e4       },
        {0x25,      0x1e,       7,      3,      0x1c        },
        {0x25,      0x1f,       10,     0,      0x7c2       },
        {0x25,      0x20,       7,      3,      0x1d        },
        {0x25,      0x21,       10,     0,      0x7ba       },
        {0x25,      0x22,       7,      3,      0x1         },
        {0x25,      0x23,       10,     0,      0x7d7       },
        {0x25,      0x24,       7,      3,      0x4         },
        {0x25,      0x25,       10,     0,      0x10        },
        {0x25,      0x26,       7,      3,      0x6         },
        {0x25,      0x27,       10,     0,      0x4c        },
        {0x25,      0x28,       7,      3,      0x3         },
        {0x25,      0x29,       10,     0,      0x6c        },
        {0x25,      0x2a,       7,      3,      0x1f        },
        {0x25,      0x2b,       10,     0,      0x59        },
        {0x25,      0x2c,       7,      3,      0x1a        },
        {0x25,      0x2d,       10,     0,      0xf         },
        {0x25,      0x2e,       7,      3,      0x18        },
        {0x25,      0x2f,       10,     0,      0x7a9       },
        {0x25,      0x30,       7,      3,      0x1b        },
        {0x25,      0x31,       10,     0,      0x753       },
        {0x25,      0x32,       7,      3,      0x1         },
        {0x25,      0x33,       10,     0,      0x742       },
        {0x25,      0x34,       7,      3,      0x8         },
        {0x25,      0x35,       10,     0,      0x79a       },
        {0x25,      0x36,       7,      3,      0xb         },
        {0x25,      0x37,       10,     0,      0x5f        },
        {0x25,      0x38,       7,      3,      0x9         },
        {0x25,      0x39,       10,     0,      0x16d       },
        {0x25,      0x3a,       7,      3,      0x0         },
        {0x25,      0x3b,       10,     0,      0x281       },
        {0x25,      0x3c,       7,      3,      0x16        },
        {0x25,      0x3d,       10,     0,      0x350       },
        {0x25,      0x3e,       7,      3,      0x10        },
        {0x25,      0x3f,       10,     0,      0x39c       },
        {0x25,      0x40,       7,      3,      0x12        },
        {0x26,      0x5,        15,     0,      0x28d9      },
        {0x26,      0x2b,       1,      1,      0x0         },
        {0x26,      0x2b,       2,      2,      0x1         },
        {0x26,      0x2b,       3,      3,      0x1         },
        {0x26,      0x32,       8,      0,      0x20        },
        {0x26,      0x34,       3,      0,      0x4         },
        {0x26,      0x35,       7,      4,      0x1         },
        {0x26,      0x4a,       2,      0,      0x1         },
        {0x26,      0x54,       5,      0,      0x3d        },
        {0x26,      0x55,       5,      0,      0x3d        },
        {0x26,      0x56,       5,      0,      0x1         },
        {0x26,      0x5a,       16,     0,      0x1982b     },
        {0x26,      0x5d,       5,      0,      0x8         },
        {0x26,      0x70,       6,      0,      0x1a        },
        {0x26,      0x71,       10,     0,      0x372       },
        {0x26,      0x77,       10,     0,      0xed        },
        {0x26,      0x79,       10,     0,      0x9         },
        {0x26,      0x87,       6,      0,      0x0         },
        {0x26,      0x8a,       7,      7,      0x1         },
        {0x26,      0xa3,       9,      0,      0x366       },
        {0x26,      0xb6,       15,     0,      0x36e       },
        {0x26,      0xd0,       12,     0,      0xadb       },
        {0x27,      0x3,        2,      2,      0x0         },
        {0x27,      0x10,       0,      0,      0x1         },
        {0x27,      0x19,       7,      0,      0x0         },
        {0x27,      0x23,       7,      0,      0x0         },
        {0x27,      0x30,       0,      0,      0x1         },
        {0x27,      0x31,       3,      0,      0x3         },
        {0x27,      0x32,       4,      0,      0x1e        },
        {0x27,      0x32,       9,      5,      0x1f        },
        {0x27,      0x36,       11,     0,      0x500       },
        {0x27,      0x37,       15,     4,      0x400       },
        {0x27,      0x3c,       11,     0,      0x33        },
        {0x27,      0x43,       0,      0,      0x0         },
        {0x27,      0x50,       0,      0,      0x1         },
        {0x27,      0x51,       3,      0,      0x3         },
        {0x27,      0x52,       4,      0,      0x1e        },
        {0x27,      0x5c,       11,     0,      0x19        },
        {0x27,      0x60,       6,      6,      0x1         },
        {0x27,      0x63,       13,     4,      0x0         },
        {0x27,      0x65,       15,     0,      0x99a       },
        {0x27,      0x6a,       13,     4,      0x0         },
        {0x27,      0x6c,       1,      1,      0x1         },
        {0x27,      0x6c,       6,      6,      0x1         },
        {0x27,      0x6d,       5,      0,      0x20        },
        {0x27,      0x6e,       15,     0,      0x4         },
        {0x27,      0x79,       7,      0,      0x0         },
        {0x27,      0x83,       7,      0,      0x0         },
        {0x27,      0xa3,       0,      0,      0x1         },
        {0x27,      0xa4,       15,     0,      0x44        },
        {0x27,      0xa6,       15,     0,      0x25e       },
        {0x27,      0xa8,       15,     0,      0x44        },
        {0x27,      0xaa,       15,     0,      0x25e       },
        {0x27,      0xac,       2,      0,      0x6         },
        {0x27,      0xac,       5,      3,      0x6         },
        {0x27,      0xac,       8,      6,      0x6         },
        {0x27,      0xad,       3,      1,      0x6         },
        {0x27,      0xad,       6,      4,      0x6         },
        {0x27,      0xad,       9,      7,      0x7         },
        {0x27,      0xae,       4,      2,      0x5         },
        {0x27,      0xae,       7,      5,      0x5         },
        {0x27,      0xaf,       5,      3,      0x5         },
        {0x27,      0xaf,       8,      6,      0x5         },
        {0x27,      0xb0,       3,      1,      0x7         },
        {0x27,      0xb2,       0,      0,      0x1         },
        {0x27,      0xb2,       1,      1,      0x1         },
        {0x27,      0xb6,       15,     0,      0xfffe      },
        {0x27,      0xb8,       5,      4,      0x0         },
        {0x27,      0xc0,       0,      0,      0x1         },
        {0x27,      0xc0,       4,      3,      0x1         },
        {0x27,      0xc0,       6,      5,      0x3         },
        {0x27,      0xc1,       2,      0,      0x3         },
        {0x27,      0xc1,       8,      6,      0x3         },
        {0x27,      0xc2,       3,      1,      0x3         },
        {0x27,      0xc3,       15,     0,      0x44        },
        {0x27,      0xc5,       15,     0,      0x44        },
        {0x27,      0xc7,       15,     0,      0x44        },
        {0x27,      0xcd,       15,     0,      0x2dc       },
        {0x28,      0x3,        7,      0,      0x20        },
        {0x28,      0x4,        3,      0,      0x6         },
        {0x28,      0x5,        9,      0,      0x3ff       },
        {0x28,      0x7,        9,      0,      0x3ff       },
        {0x28,      0x9,        9,      0,      0x148       },
        {0x28,      0xc,        3,      0,      0xf         },
        {0x28,      0xd,        11,     0,      0x10d       },
        {0x28,      0xf,        11,     0,      0x9         },
        {0x28,      0x1d,       1,      0,      0x3         },
        {0x28,      0x1d,       4,      2,      0x2         },
        {0x28,      0x1d,       7,      5,      0x2         },
        {0x28,      0x1e,       9,      0,      0x218       },
        {0x28,      0x20,       9,      0,      0x218       },
        {0x28,      0x22,       9,      0,      0x42        },
        {0x28,      0x24,       5,      0,      0x2         },
        {0x28,      0x24,       6,      6,      0x0         },
        {0x28,      0x25,       9,      0,      0x10d       },
        {0x28,      0x26,       11,     2,      0x2b        },
        {0x28,      0x28,       9,      0,      0x4a        },
        {0x28,      0x2a,       9,      0,      0x3ff       },
        {0x28,      0x2c,       9,      0,      0x3ff       },
        {0x28,      0x2e,       9,      0,      0x0         },
        {0x28,      0x30,       0,      0,      0x1         },
        {0x28,      0x31,       15,     0,      0xe         },
        {0x28,      0x33,       7,      0,      0x20        },
        {0x28,      0x34,       4,      1,      0xa         },
        {0x28,      0x39,       31,     2,      0x15bd9bd3  },
        {0x28,      0x3d,       3,      0,      0x0         },
        {0x28,      0x46,       5,      5,      0x0         },
        {0x28,      0x54,       3,      0,      0x5         },
        {0x28,      0x54,       7,      4,      0x5         },
        {0x30,      0xf,        7,      0,      0x88        },
        {0x30,      0x10,       7,      0,      0x20        },
        {0x31,      0x6,        7,      0,      0x18        },
        {0x31,      0x9,        7,      0,      0x90        },
        {0x31,      0xa,        7,      0,      0xd0        },
        {0x31,      0xb,        7,      0,      0x3         },
        {0x31,      0xf,        7,      0,      0x90        },
        {0x31,      0x10,       7,      0,      0x5f        },
        {0x31,      0x11,       7,      0,      0xaf        },
        {0x31,      0x13,       7,      0,      0x45        },
        {0x31,      0x14,       7,      0,      0xc0        },
        {0x31,      0x15,       7,      0,      0x51        },
        {0x31,      0x16,       7,      0,      0x2         },
        {0x31,      0x1a,       7,      0,      0x10        },
        {0x32,      0x6,        7,      0,      0x14        },
        {0x32,      0x7,        7,      0,      0x84        },
        {0x32,      0x8,        7,      0,      0x10        },
        {0x32,      0x9,        7,      0,      0xf0        },
        {0x32,      0xa,        7,      0,      0x49        },
        {0x32,      0xb,        7,      0,      0x2         },
        {0x32,      0xf,        7,      0,      0x90        },
        {0x32,      0x10,       7,      0,      0x3f        },
        {0x32,      0x12,       7,      0,      0x2a        },
        {0x32,      0x13,       7,      0,      0x45        },
        {0x33,      0xf,        7,      0,      0x90        },
        {0x33,      0x10,       7,      0,      0x37        },
        {0x33,      0x12,       7,      0,      0x2a        },
        {0x33,      0x13,       7,      0,      0x45        },
        {0x34,      0xf,        7,      0,      0x90        },
        {0x34,      0x10,       7,      0,      0x3f        },
        {0x34,      0x12,       7,      0,      0x2a        },
        {0x34,      0x13,       7,      0,      0x45        },
        {0x35,      0x1,        7,      0,      0x45        },
        {0x35,      0x2,        7,      0,      0x50        },
        {0x35,      0x3,        7,      0,      0x4         },
        {0x35,      0x5,        7,      0,      0x60        },
        {0x35,      0x7,        7,      0,      0x58        },
        {0x35,      0xf,        7,      0,      0x90        },
        {0x35,      0x10,       7,      0,      0x5f        },
        {0x35,      0x12,       7,      0,      0x2a        },
        {0x35,      0x13,       7,      0,      0x45        },
        {0x35,      0x14,       7,      0,      0xe0        },
        {0x35,      0x15,       7,      0,      0x11        },
        {0x35,      0x16,       7,      0,      0x0         },
        {0x35,      0x1a,       7,      0,      0x10        },
        {0x36,      0xf,        7,      0,      0x90        },
        {0x36,      0x10,       7,      0,      0x5f        },
        {0x36,      0x13,       7,      0,      0xc5        },
        {0x37,      0x6,        7,      0,      0x14        },
        {0x37,      0x9,        7,      0,      0x98        },
        {0x37,      0xa,        7,      0,      0x3a        },
        {0x37,      0xf,        7,      0,      0x90        },
        {0x37,      0x11,       7,      0,      0xa9        },
        {0x37,      0x12,       7,      0,      0x2a        },
        {0x37,      0x13,       7,      0,      0x45        },
        {0x37,      0x14,       7,      0,      0xc1        },
        {0x38,      0x6,        7,      0,      0x20        },
        {0x38,      0x7,        7,      0,      0x0         },
        {0x38,      0x8,        7,      0,      0x0         },
        {0x38,      0x9,        7,      0,      0x0         },
        {0x38,      0xa,        7,      0,      0x0         },
        {0x38,      0xf,        7,      0,      0x0         },
        {0x38,      0x10,       7,      0,      0x20        },
        {0x38,      0x11,       7,      0,      0x0         },
        {0x38,      0x12,       7,      0,      0x0         },
        {0x38,      0x13,       7,      0,      0x0         },
        {0x38,      0x14,       7,      0,      0x0         },
        {0x38,      0x15,       7,      0,      0x0         },
        {0x38,      0x16,       7,      0,      0x0         },
        {0x39,      0x6,        7,      0,      0x90        },
        {0x39,      0x8,        7,      0,      0x1c        },
        {0x39,      0x9,        7,      0,      0x20        },
        {0x39,      0xa,        7,      0,      0xa1        },
        {0x39,      0xb,        7,      0,      0x7         },
        {0x39,      0xf,        7,      0,      0x90        },
        {0x39,      0x10,       7,      0,      0xbf        },
        {0x39,      0x14,       7,      0,      0xec        },
        {0x39,      0x15,       7,      0,      0x70        },
        {0x3a,      0x9,        7,      0,      0x90        },
        {0x3a,      0xa,        7,      0,      0xd0        },
        {0x3a,      0xb,        7,      0,      0x3         },
        {0x3a,      0xf,        7,      0,      0x90        },
        {0x3a,      0x10,       7,      0,      0x7f        },
        {0x3b,      0x9,        7,      0,      0xa0        },
        {0x3b,      0xa,        7,      0,      0x86        },
        {0x3b,      0xb,        7,      0,      0x1         },
        {0x3b,      0xf,        7,      0,      0x90        },
        {0x3b,      0x10,       7,      0,      0x9f        },
        {0x3c,      0x4,        7,      0,      0xd1        },
        {0x3c,      0x5,        7,      0,      0x48        },
        {0x3c,      0x6,        7,      0,      0x51        },
        {0x3c,      0x7,        7,      0,      0x9b        },
        {0x3c,      0x8,        7,      0,      0x1d        },
        {0x3c,      0x9,        7,      0,      0xf0        },
        {0x3c,      0xa,        7,      0,      0x49        },
        {0x3c,      0xb,        7,      0,      0x2         },
        {0x3c,      0xf,        7,      0,      0x90        },
        {0x3c,      0x10,       7,      0,      0x9f        },
        {0x3d,      0x6,        7,      0,      0x14        },
        {0x3d,      0xf,        7,      0,      0x90        },
        {0x3d,      0x10,       7,      0,      0x9f        },
        {0x3d,      0x11,       7,      0,      0xaa        },
        {0x3e,      0xf,        7,      0,      0x90        },
        {0x3e,      0x10,       7,      0,      0x7f        },
        {0x3f,      0x1,        7,      0,      0x8f        },
        {0x3f,      0x2,        7,      0,      0xf0        },
        {0x3f,      0xf,        7,      0,      0x90        },
        {0x3f,      0x10,       7,      0,      0x7f        },
        {0x40,      0x6,        7,      0,      0x2d        },
        {0x40,      0x7,        7,      0,      0x8c        },
        {0x40,      0xf,        7,      0,      0x90        },
        {0x40,      0x10,       7,      0,      0x7f        },
        {0x41,      0x9,        7,      0,      0xf4        },
        {0x41,      0xa,        7,      0,      0x1         },
        {0x41,      0xb,        7,      0,      0x0         },
        {0x41,      0xf,        7,      0,      0x90        },
        {0x41,      0x10,       7,      0,      0x9f        },
        {0x42,      0x1,        7,      0,      0x92        },
        {0x42,      0x2,        7,      0,      0x64        },
        {0x42,      0x3,        7,      0,      0x4e        },
        {0x42,      0x4,        7,      0,      0x92        },
        {0x42,      0x5,        7,      0,      0x4         },
        {0x42,      0x6,        7,      0,      0x8a        },
        {0x42,      0x7,        7,      0,      0xef        },
        {0x42,      0x8,        7,      0,      0x7c        },
        {0x42,      0x9,        7,      0,      0x60        },
        {0x42,      0xa,        7,      0,      0x1         },
        {0x42,      0xb,        7,      0,      0x0         },
        {0x42,      0xe,        7,      0,      0x1         },
        {0x42,      0xf,        7,      0,      0xb0        },
        {0x42,      0x10,       7,      0,      0xbd        },
        {0x42,      0x11,       7,      0,      0x3e        },
        {0x42,      0x12,       7,      0,      0xd8        },
        {0x42,      0x13,       7,      0,      0xc2        },
        {0x42,      0x14,       7,      0,      0x31        },
        {0x42,      0x16,       7,      0,      0x42        },
        {0x42,      0x17,       7,      0,      0xaa        },
        {0x42,      0x18,       7,      0,      0xf8        },
        {0x42,      0x19,       7,      0,      0x9f        },
        {0x42,      0x1a,       7,      0,      0x3         },
        {0x43,      0x6,        7,      0,      0x99        },
        {0x43,      0x9,        7,      0,      0x3a        },
        {0x43,      0xa,        7,      0,      0x0         },
        {0x43,      0xf,        7,      0,      0xb0        },
        {0x43,      0x10,       7,      0,      0xbd        },
        {0x43,      0x11,       7,      0,      0x3e        },
        {0x43,      0x12,       7,      0,      0xd8        },
        {0x43,      0x13,       7,      0,      0xc2        },
        {0x43,      0x14,       7,      0,      0x31        },
        {0x43,      0x16,       7,      0,      0x42        },
        {0x43,      0x17,       7,      0,      0xaa        },
        {0x43,      0x18,       7,      0,      0xf8        },
        {0x43,      0x19,       7,      0,      0x9f        },
        {0x43,      0x1a,       7,      0,      0x3         },
        {0x44,      0xf,        7,      0,      0x90        },
        {0x44,      0x10,       7,      0,      0xbf        },
        {0x44,      0x14,       7,      0,      0x71        },
        {0x44,      0x15,       7,      0,      0xec        },
        {0x45,      0xf,        7,      0,      0x90        },
        {0x45,      0x10,       7,      0,      0xbf        },
        {0x45,      0x15,       7,      0,      0xec        },
        {0x46,      0xf,        7,      0,      0x90        },
        {0x46,      0x10,       7,      0,      0xbf        },
        {0x46,      0x15,       7,      0,      0xac        },
        {0x47,      0xc,        7,      0,      0x44        },
        {0x47,      0xf,        7,      0,      0x90        },
        {0x47,      0x10,       7,      0,      0xbf        },
        {0x47,      0x15,       7,      0,      0x2c        },
        {0x48,      0xf,        7,      0,      0x90        },
        {0x48,      0x10,       7,      0,      0xbf        },
        {0x48,      0x15,       7,      0,      0x6c        },
        {0x49,      0xf,        7,      0,      0x90        },
        {0x49,      0x10,       7,      0,      0xbf        },
        {0x49,      0x15,       7,      0,      0x6c        },
        {0x4a,      0xf,        7,      0,      0x90        },
        {0x4a,      0x10,       7,      0,      0xbf        },
        {0x4a,      0x15,       7,      0,      0x6c        },
        {0x4b,      0xf,        7,      0,      0x90        },
        {0x4b,      0x10,       7,      0,      0xbf        },
        {0x4b,      0x15,       7,      0,      0x6c        },
        {0x4c,      0xf,        7,      0,      0x90        },
        {0x4c,      0x10,       7,      0,      0xbf        },
        {0x4c,      0x15,       7,      0,      0x6c        },
        {0x4d,      0xf,        7,      0,      0x90        },
        {0x4d,      0x10,       7,      0,      0xbf        },
        {0x4d,      0x15,       7,      0,      0x6c        },
        {0x4e,      0x5,        7,      0,      0x67        },
        {0x4e,      0xf,        7,      0,      0x90        },
        {0x4e,      0x10,       7,      0,      0xbf        },
        {0x4e,      0x15,       7,      0,      0x6c        },
        {0x4f,      0x6,        7,      0,      0x99        },
        {0x4f,      0xf,        7,      0,      0x90        },
        {0x4f,      0x10,       7,      0,      0xbf        },
        {0x4f,      0x15,       7,      0,      0x6c        },
        {0x50,      0x7,        7,      0,      0x27        },
        {0x50,      0xf,        7,      0,      0xb0        },
        {0x50,      0x10,       7,      0,      0xbf        },
        {0x50,      0x11,       7,      0,      0x3e        },
        {0x50,      0x12,       7,      0,      0xd8        },
        {0x50,      0x14,       7,      0,      0x31        },
        {0x51,      0x2,        7,      0,      0x8         },
        {0x51,      0x3,        7,      0,      0x8a        },
        {0x51,      0x4,        7,      0,      0x65        },
        {0x51,      0x5,        7,      0,      0x59        },
        {0x51,      0x6,        7,      0,      0x96        },
        {0x51,      0x8,        7,      0,      0x54        },
        {0x51,      0xf,        7,      0,      0xb0        },
        {0x51,      0x10,       7,      0,      0xbf        },
        {0x51,      0x11,       7,      0,      0x3e        },
        {0x51,      0x12,       7,      0,      0xd8        },
        {0x51,      0x14,       7,      0,      0x31        },
        {0x52,      0xf,        7,      0,      0xb0        },
        {0x52,      0x10,       7,      0,      0xbf        },
        {0x52,      0x11,       7,      0,      0x3e        },
        {0x52,      0x12,       7,      0,      0xd8        },
        {0x52,      0x14,       7,      0,      0x31        },
        {0x53,      0x6,        7,      0,      0x26        },
        {0x53,      0xf,        7,      0,      0x90        },
        {0x53,      0x10,       7,      0,      0xbf        },
        {0x53,      0x14,       7,      0,      0xc1        },
        {0x53,      0x15,       7,      0,      0x50        },
        {0x54,      0x2,        7,      0,      0x45        },
        {0x54,      0x3,        7,      0,      0x51        },
        {0x54,      0x4,        7,      0,      0xc7        },
        {0x54,      0x5,        7,      0,      0x71        },
        {0x54,      0x6,        7,      0,      0x1c        },
        {0x54,      0x7,        7,      0,      0xad        },
        {0x54,      0x8,        7,      0,      0x59        },
        {0x54,      0x9,        7,      0,      0x20        },
        {0x54,      0xa,        7,      0,      0xa1        },
        {0x54,      0xb,        7,      0,      0x7         },
        {0x54,      0xf,        7,      0,      0x90        },
        {0x54,      0x10,       7,      0,      0xbf        },
        {0x54,      0x13,       7,      0,      0xc5        },
        {0x54,      0x14,       7,      0,      0xec        },
        {0x54,      0x15,       7,      0,      0x70        },
        {0x54,      0x1a,       7,      0,      0x1c        },
        {0x55,      0x1,        7,      0,      0xc7        },
        {0x55,      0x2,        7,      0,      0x71        },
        {0x55,      0x3,        7,      0,      0x1c        },
        {0x55,      0x4,        7,      0,      0xc7        },
        {0x55,      0x5,        7,      0,      0x71        },
        {0x55,      0x6,        7,      0,      0x1c        },
        {0x55,      0x7,        7,      0,      0x87        },
        {0x55,      0x8,        7,      0,      0x89        },
        {0x55,      0xf,        7,      0,      0x90        },
        {0x55,      0x10,       7,      0,      0xbf        },
        {0x55,      0x11,       7,      0,      0xaa        },
        {0x55,      0x12,       7,      0,      0xdc        },
        {0x55,      0x13,       7,      0,      0xc2        },
        {0x55,      0x15,       7,      0,      0x6c        },
        {0x55,      0x16,       7,      0,      0x5a        },
        {0x55,      0x17,       7,      0,      0xb2        },
        {0x55,      0x18,       7,      0,      0xfc        },
        {0x55,      0x19,       7,      0,      0x9f        },
        {0x55,      0x1a,       7,      0,      0x3         },
        {0x56,      0x1,        7,      0,      0x26        },
        {0x56,      0x2,        7,      0,      0x65        },
        {0x56,      0x3,        7,      0,      0x52        },
        {0x56,      0x4,        7,      0,      0x26        },
        {0x56,      0x5,        7,      0,      0x25        },
        {0x56,      0x6,        7,      0,      0x51        },
        {0x56,      0x7,        7,      0,      0x27        },
        {0x56,      0x8,        7,      0,      0x42        },
        {0x56,      0x9,        7,      0,      0xf0        },
        {0x56,      0xa,        7,      0,      0x49        },
        {0x56,      0xb,        7,      0,      0x2         },
        {0x56,      0xf,        7,      0,      0xb0        },
        {0x56,      0x10,       7,      0,      0xbd        },
        {0x56,      0x11,       7,      0,      0x3e        },
        {0x56,      0x12,       7,      0,      0xd8        },
        {0x56,      0x13,       7,      0,      0xc2        },
        {0x56,      0x14,       7,      0,      0x31        },
        {0x56,      0x15,       7,      0,      0x6c        },
        {0x56,      0x16,       7,      0,      0x5a        },
        {0x56,      0x17,       7,      0,      0xaa        },
        {0x56,      0x18,       7,      0,      0xf8        },
        {0x56,      0x19,       7,      0,      0x9f        },
        {0x56,      0x1a,       7,      0,      0x3         },
        {0x57,      0xf,        7,      0,      0x90        },
        {0x57,      0x10,       7,      0,      0xbf        },
        {0x57,      0x15,       7,      0,      0x2c        },
        {0x0,       0x2,        7,      0,      0x0         },
        {0x0,       0x4,        0,      0,      0x0         },
        {0x0,       0x4,        1,      0,      0x0         },
        {0x0,       0x5,        1,      0,      0x0         },
        {0x0,       0x6,        2,      1,      0x1         },
        {0x0,       0x6,        5,      1,      0x1         },
        {0x0,       0x6,        5,      1,      0x1         },
        {0x0,       0x7,        4,      0,      0x0         },
        {0x0,       0x8,        3,      1,      0x2         },
        {0x0,       0x8,        7,      5,      0x6         },
        {0x0,       0xb,        5,      0,      0x20        },
        {0x0,       0xc,        5,      0,      0x24        },
        {0x0,       0xd,        5,      0,      0x24        },
        {0x0,       0xe,        3,      0,      0x0         },
        {0x0,       0xf,        7,      0,      0x0         },
        {0x0,       0x10,       7,      0,      0x0         },
        {0x0,       0x11,       7,      0,      0x0         },
        {0x0,       0x12,       7,      0,      0x0         },
        {0x0,       0x15,       7,      0,      0x0         },
        {0x0,       0x16,       3,      2,      0x0         },
        {0x0,       0x17,       5,      0,      0x0         },
        {0x0,       0x18,       0,      0,      0x0         },
        {0x0,       0x1f,       1,      0,      0x0         },
        {0x0,       0x50,       0,      0,      0x0         },
        {0x0,       0x51,       4,      0,      0x0         },
        {0x0,       0x52,       7,      0,      0x0         },
        {0x0,       0x53,       7,      0,      0x0         },
        {0x0,       0x54,       7,      0,      0x0         },
        {0x0,       0x55,       7,      0,      0x0         },
        {0x0,       0x58,       0,      0,      0x0         },
        {0x0,       0x59,       1,      0,      0x0         },
        {0x0,       0x5a,       7,      0,      0x0         },
        {0x0,       0x5b,       7,      0,      0x0         },
        {0x0,       0x5c,       7,      0,      0x0         },
        {0x0,       0x5d,       7,      0,      0x0         },
        {0x0,       0x60,       7,      0,      0x65        },
        {0x0,       0x61,       7,      0,      0x93        },
        {0x0,       0x62,       7,      0,      0xa0        },
        {0x0,       0x63,       7,      0,      0x8d        },
        {0x0,       0x64,       7,      0,      0x75        },
        {0x0,       0x65,       7,      0,      0x64        },
        {0x0,       0x66,       7,      0,      0x42        },
        {0x0,       0x67,       7,      0,      0x1         },
        {0x0,       0x4,        1,      0,      0x2         },
    };


    static const TS_INTERFACE_INIT_TABLE_ENTRY TsInterfaceInitTable[RTD2885_ATSC_TS_INTERFACE_INIT_TABLE_LEN] =
    {
        // RegBitName,              WritingValue for {Parallel, Serial}
        {ATSC_SERIAL,               {0x0,   0x1}},
        {ATSC_CDIV_PH0,             {0x5,   0x1}},
        {ATSC_CDIV_PH1,             {0x5,   0x2}},
    };


    int i;

    int TsInterfaceMode;

    unsigned char PageNo;
    unsigned char RegStartAddr;
    unsigned char Msb;
    unsigned char Lsb;
    unsigned long WritingValue;



    // Get TS interface mode.
    TsInterfaceMode = pDemod->TsInterfaceMode;

    // Initialize demod with register initializing table.
    for(i = 0; i < RTD2885_ATSC_INIT_REG_TABLE_LEN; i++)
    {
        // Get all information from each register initializing entry.
        PageNo       = InitRegTable[i].PageNo;
        RegStartAddr = InitRegTable[i].RegStartAddr;
        Msb          = InitRegTable[i].Msb;
        Lsb          = InitRegTable[i].Lsb;
        WritingValue = InitRegTable[i].WritingValue;

        // Set register page number.
        if(pDemod->SetRegPage(pDemod, PageNo) != FUNCTION_SUCCESS)
            goto error_status_set_demod_registers;

        // Set register mask bits.
        if(pDemod->SetRegMaskBits(pDemod, RegStartAddr, Msb, Lsb, WritingValue) != FUNCTION_SUCCESS)
            goto error_status_set_demod_registers;
    }


    // Initialize demod registers according to the TS interface initializing table.
    for(i = 0; i < RTD2885_ATSC_TS_INTERFACE_INIT_TABLE_LEN; i++)
    {
        if(pDemod->SetRegBitsWithPage(pDemod, TsInterfaceInitTable[i].RegBitName,
            TsInterfaceInitTable[i].WritingValue[TsInterfaceMode]) != FUNCTION_SUCCESS)
            goto error_status_set_demod_registers;
    }


    // Enable MPEG output.
    if(pDemod->SetRegBitsWithPage(pDemod, ATSC_OPT_M_OEN, 0x1) != FUNCTION_SUCCESS)
        goto error_status_set_demod_mpeg_output_enable;
    
    
    return FUNCTION_SUCCESS;


error_status_set_demod_mpeg_output_enable:
error_status_set_demod_registers:
    return FUNCTION_ERROR;
}





/**

@see   ATSC_DEMOD_FP_SET_SYMBOL_RATE_HZ

*/
int
rtd2885_atsc_SetSymbolRateHz(
    ATSC_DEMOD_MODULE *pDemod,
    unsigned long SymbolRateHz
    )
{
    unsigned long CrystalFreqHz;
    
    MPI MpiSymbolRateHz, MpiCrystalFreqHz, MpiConst, MpiVar, MpiNone;
    unsigned long CrConstInc, TrPhaseInc;



    // Get demod crystal frequency in Hz.
    pDemod->GetCrystalFreqHz(pDemod, &CrystalFreqHz);


    // Initialize MPI variables.
    MpiSetValue(&MpiCrystalFreqHz, (long)CrystalFreqHz);
    MpiSetValue(&MpiSymbolRateHz, (long)SymbolRateHz);
    MpiSetValue(&MpiConst, 1);


    // Calculate CR_CONST_INC value.
    // Note: Original formula:   CR_CONST_INC = round( (SymbolRateHz / CrystalFreqHz) * pow(2, 18) )
    //       Adjusted formula:   CR_CONST_INC = floor( ( (SymbolRateHz << 19) / CrystalFreqHz + 1 ) >> 1)
    MpiLeftShift(&MpiVar, MpiSymbolRateHz, 19);
    MpiDiv(&MpiVar, &MpiNone, MpiVar, MpiCrystalFreqHz);
    MpiAdd(&MpiVar, MpiVar, MpiConst);
    MpiRightShift(&MpiVar, MpiVar, 1);

    MpiGetValue(MpiVar, (long *)&CrConstInc);


    // Set CR_CONST_INC with calculated value.
    // Note: The function SetRegBitsWithPage() will set register page automatically.
    if(pDemod->SetRegBitsWithPage(pDemod, ATSC_CR_CONST_INC, CrConstInc) != FUNCTION_SUCCESS)
        goto error_status_set_demod_registers;


    // Calculate TR_PHASE_INC value.
    // Note: Original formula:   TR_PHASE_INC = round( (CrystalFreqHz / SymbolRateHz) * pow(2, 24) )
    //       Adjusted formula:   TR_PHASE_INC = floor( ( (CrystalFreqHz << 25) / SymbolRateHz + 1 ) >> 1)
    MpiLeftShift(&MpiVar, MpiCrystalFreqHz, 25);
    MpiDiv(&MpiVar, &MpiNone, MpiVar, MpiSymbolRateHz);
    MpiAdd(&MpiVar, MpiVar, MpiConst);
    MpiRightShift(&MpiVar, MpiVar, 1);

    MpiGetValue(MpiVar, (long *)&TrPhaseInc);


    // Set TR_PHASE_INC with calculated value.
    // Note: The function SetRegBitsWithPage() will set register page automatically.
    if(pDemod->SetRegBitsWithPage(pDemod, ATSC_TR_PHASE_INC, TrPhaseInc) != FUNCTION_SUCCESS)
        goto error_status_set_demod_registers;


    // Set demod symbol rate parameter.
    pDemod->SymbolRateHz      = SymbolRateHz;
    pDemod->IsSymbolRateHzSet = YES;


    return FUNCTION_SUCCESS;


error_status_set_demod_registers:
    return FUNCTION_ERROR;
}





/**

@see   ATSC_DEMOD_FP_SET_IF_FREQ_HZ

*/
int
rtd2885_atsc_SetIfFreqHz(
    ATSC_DEMOD_MODULE *pDemod,
    unsigned long IfFreqHz
    )
{
    unsigned long CrystalFreqHz;
    long AdjustedIfFreqHz;

    MPI MpiAdjustedIfFreqHz, MpiCrystalFreqHz, MpiConst, MpiVar, MpiNone;
    unsigned long DdcPhaseInc;



    // Get demod crystal frequency in Hz.
    pDemod->GetCrystalFreqHz(pDemod, &CrystalFreqHz);


    // Get adjusted IF frequency.
    // Note: Step 1. AdjustedIfFreqHz = IfFreqHz - CrystalFreqHz * 2 * N
    //               a. N is an integer and N >= 0
    //               b. (- CrystalFreqHz) < AdjustedIfFreqHz <= CrystalFreqHz
    //       Step 2. AdjustedIfFreqHz = abs(AdjustedIfFreqHz)
    AdjustedIfFreqHz = IfFreqHz;

    while(AdjustedIfFreqHz > (long)CrystalFreqHz)
        AdjustedIfFreqHz -= CrystalFreqHz * 2;

    if(AdjustedIfFreqHz < 0)
        AdjustedIfFreqHz = - AdjustedIfFreqHz;


    // Calculate DDC_PHASE_INC value.
    // Note: Original formula:   DDC_PHASE_INC = round( (AdjustedIfFreqHz / CrystalFreqHz) * pow(2, 20) )
    //       Adjusted formula:   DDC_PHASE_INC = floor( ( (AdjustedIfFreqHz << 21) / CrystalFreqHz + 1 ) >> 1)
    MpiSetValue(&MpiCrystalFreqHz, (long)CrystalFreqHz);
    MpiSetValue(&MpiAdjustedIfFreqHz, AdjustedIfFreqHz);
    MpiSetValue(&MpiConst, 1);

    MpiLeftShift(&MpiVar, MpiAdjustedIfFreqHz, 21);
    MpiDiv(&MpiVar, &MpiNone, MpiVar, MpiCrystalFreqHz);
    MpiAdd(&MpiVar, MpiVar, MpiConst);
    MpiRightShift(&MpiVar, MpiVar, 1);

    MpiGetValue(MpiVar, (long *)&DdcPhaseInc);


    // Set DDC_PHASE_INC with calculated value.
    // Note: The function SetRegBitsWithPage() will set register page automatically.
    if(pDemod->SetRegBitsWithPage(pDemod, ATSC_DDC_PHASE_INC, DdcPhaseInc) != FUNCTION_SUCCESS)
        goto error_status_set_demod_registers;


    // Set demod IF frequnecy parameter.
    pDemod->IfFreqHz      = IfFreqHz;
    pDemod->IsIfFreqHzSet = YES;


    return FUNCTION_SUCCESS;


error_status_set_demod_registers:
    return FUNCTION_ERROR;
}





/**

@see   ATSC_DEMOD_FP_SET_SPECTRUM_MODE

*/
int
rtd2885_atsc_SetSpectrumMode(
    ATSC_DEMOD_MODULE *pDemod,
    int SpectrumMode
    )
{
    unsigned long SpecInvOn;


    // Determine SPEC_INV_ON according to spectrum mode.
    switch(SpectrumMode)
    {
        case SPECTRUM_NORMAL:

            SpecInvOn = 0x0;

            break;

        default:
        case SPECTRUM_INVERSE:

            SpecInvOn = 0x1;

            break;
    }


    // Set SPEC_INV_ON with determined value.
    // Note: The function SetRegBitsWithPage() will set register page automatically.
    if(pDemod->SetRegBitsWithPage(pDemod, ATSC_SPEC_INV_ON, SpecInvOn) != FUNCTION_SUCCESS)
        goto error_status_set_demod_registers;


    // Set demod spectrum mode parameter.
    pDemod->SpectrumMode      = SpectrumMode;
    pDemod->IsSpectrumModeSet = YES;


    return FUNCTION_SUCCESS;


error_status_set_demod_registers:
    return FUNCTION_ERROR;
}





/**

@see   ATSC_DEMOD_FP_GET_RF_AGC

*/
int
rtd2885_atsc_GetRfAgc(
    ATSC_DEMOD_MODULE *pDemod,
    long *pRfAgc
    )
{
    unsigned long RfAgcBinary;



    // Set RO_STROBE with 0x1, then set RO_STROBE with 0x0.
    // Note: RO_STROBE is a write-only register.
    if(pDemod->SetRegBitsWithPage(pDemod, ATSC_RO_STROBE, 0x1) != FUNCTION_SUCCESS)
        goto error_status_set_demod_registers;

    if(pDemod->SetRegBitsWithPage(pDemod, ATSC_RO_STROBE, 0x0) != FUNCTION_SUCCESS)
        goto error_status_set_demod_registers;


    // Get RF AGC binary value from RO_RF_AAGC.
    // Note: The function GetRegBitsWithPage() will set register page automatically.
    if(pDemod->GetRegBitsWithPage(pDemod, ATSC_RO_RF_AAGC, &RfAgcBinary) != FUNCTION_SUCCESS)
        goto error_status_get_demod_registers;


    // Convert RF AGC binary value to signed integer.
    *pRfAgc = BinToSignedInt(RfAgcBinary, RTD2885_ATSC_RF_AGC_VALUE_BIT_NUM);


    return FUNCTION_SUCCESS;


error_status_get_demod_registers:
error_status_set_demod_registers:
    return FUNCTION_ERROR;
}




/**

@see   ATSC_DEMOD_FP_GET_IF_AGC

*/
int
rtd2885_atsc_GetIfAgc(
    ATSC_DEMOD_MODULE *pDemod,
    long *pIfAgc
    )
{
    unsigned long IfAgcBinary;



    // Set RO_STROBE with 0x1, then set RO_STROBE with 0x0.
    // Note: RO_STROBE is a write-only register.
    if(pDemod->SetRegBitsWithPage(pDemod, ATSC_RO_STROBE, 0x1) != FUNCTION_SUCCESS)
        goto error_status_set_demod_registers;

    if(pDemod->SetRegBitsWithPage(pDemod, ATSC_RO_STROBE, 0x0) != FUNCTION_SUCCESS)
        goto error_status_set_demod_registers;


    // Get IF AGC binary value from RO_IF_AAGC.
    // Note: The function GetRegBitsWithPage() will set register page automatically.
    if(pDemod->GetRegBitsWithPage(pDemod, ATSC_RO_IF_AAGC, &IfAgcBinary) != FUNCTION_SUCCESS)
        goto error_status_get_demod_registers;


    // Convert IF AGC binary value to signed integer.
    *pIfAgc = BinToSignedInt(IfAgcBinary, RTD2885_ATSC_IF_AGC_VALUE_BIT_NUM);


    return FUNCTION_SUCCESS;


error_status_get_demod_registers:
error_status_set_demod_registers:
    return FUNCTION_ERROR;
}





/**

@see   ATSC_DEMOD_FP_GET_DI_AGC

*/
int
rtd2885_atsc_GetDiAgc(
    ATSC_DEMOD_MODULE *pDemod,
    unsigned long *pDiAgc
    )
{
    // Set RO_STROBE with 0x1, then set RO_STROBE with 0x0.
    // Note: RO_STROBE is a write-only register.
    if(pDemod->SetRegBitsWithPage(pDemod, ATSC_RO_STROBE, 0x1) != FUNCTION_SUCCESS)
        goto error_status_set_demod_registers;

    if(pDemod->SetRegBitsWithPage(pDemod, ATSC_RO_STROBE, 0x0) != FUNCTION_SUCCESS)
        goto error_status_set_demod_registers;


    // Get digital AGC value from RO_DAGC_LEVEL_Q.
    // Note: The function GetRegBitsWithPage() will set register page automatically.
    if(pDemod->GetRegBitsWithPage(pDemod, ATSC_RO_DAGC_LEVEL_Q, pDiAgc) != FUNCTION_SUCCESS)
        goto error_status_get_demod_registers;


    return FUNCTION_SUCCESS;


error_status_get_demod_registers:
error_status_set_demod_registers:
    return FUNCTION_ERROR;
}





/**

@see   ATSC_DEMOD_FP_GET_TR_OFFSET_PPM

*/
int
rtd2885_atsc_GetTrOffsetPpm(
    ATSC_DEMOD_MODULE *pDemod,
    long *pTrOffsetPpm
    )
{
    unsigned long SymbolRateHz;
    unsigned long CrystalFreqHz;

    unsigned long TrOffsetN1Binary;
    long          TrOffsetN1Int;

    unsigned long TrOffsetN2Binary;
    long          TrOffsetN2Int;

    MPI MpiSymbolRateHz, MpiCrystalFreqHz, MpiTrOffsetN1Int, MpiTrOffsetN2Int;
    MPI MpiVarN1, MpiVarN2;
    MPI MpiConst, MpiVar0, MpiVar1, MpiNone;



    // Get demod symbol rate in Hz.
    if(pDemod->GetSymbolRateHz(pDemod, &SymbolRateHz) != FUNCTION_SUCCESS)
        goto error_status_get_demod_symbol_rate;


    // Get demod crystal frequency in Hz.
    pDemod->GetCrystalFreqHz(pDemod, &CrystalFreqHz);


    // Set RO_STROBE with 0x1, then set RO_STROBE with 0x0.
    // Note: RO_STROBE is a write-only register.
    if(pDemod->SetRegBitsWithPage(pDemod, ATSC_RO_STROBE, 0x1) != FUNCTION_SUCCESS)
        goto error_status_set_demod_registers;

    if(pDemod->SetRegBitsWithPage(pDemod, ATSC_RO_STROBE, 0x0) != FUNCTION_SUCCESS)
        goto error_status_set_demod_registers;


    // Get TR offset N1 binary value from RO_TR_OFFSET_N1 register bits.
    // Note: The function GetRegBitsWithPage() will set register page automatically.
    if(pDemod->GetRegBitsWithPage(pDemod, ATSC_RO_TR_OFFSET_N1, &TrOffsetN1Binary) != FUNCTION_SUCCESS)
        goto error_status_get_demod_registers;

    // Convert TR offset N1 binary value to signed integer.
    TrOffsetN1Int = BinToSignedInt(TrOffsetN1Binary, RTD2885_ATSC_TR_OFFSET_N1_BIT_NUM);


    // Get TR offset binary N2 value from RO_TR_OFFSET_N2 register bits.
    // Note: The function GetRegBitsWithPage() will set register page automatically.
    if(pDemod->GetRegBitsWithPage(pDemod, ATSC_RO_TR_OFFSET_N2, &TrOffsetN2Binary) != FUNCTION_SUCCESS)
        goto error_status_get_demod_registers;

    // Convert TR offset N2 binary value to signed integer.
    TrOffsetN2Int = BinToSignedInt(TrOffsetN2Binary, RTD2885_ATSC_TR_OFFSET_N2_BIT_NUM);


    // Get TR offset in ppm.
    // Note: Original formula:   TrOffsetPpm = TrOffsetN1 - TrOffsetN2
    //                           TrOffsetN1  = (TrOffsetN1Int * SymbolRateHz * 1000000) / (CrystalFreqHz * pow(2, 19))
    //                           TrOffsetN2  = (TrOffsetN2Int * SymbolRateHz * 1000000) / (CrystalFreqHz * pow(2, 23))
    //       Adjusted formula:   TrOffsetPpm = TrOffsetN1 - TrOffsetN2
    //                           TrOffsetN1  = (TrOffsetN1Int * SymbolRateHz * 1000000) / (CrystalFreqHz << 19)
    //                           TrOffsetN2  = (TrOffsetN2Int * SymbolRateHz * 1000000) / (CrystalFreqHz << 23)
    MpiSetValue(&MpiSymbolRateHz,  SymbolRateHz);
    MpiSetValue(&MpiCrystalFreqHz, CrystalFreqHz);
    MpiSetValue(&MpiTrOffsetN1Int, TrOffsetN1Int);
    MpiSetValue(&MpiTrOffsetN2Int, TrOffsetN2Int);
    MpiSetValue(&MpiConst,         1000000);

    MpiMul(&MpiVar0, MpiTrOffsetN1Int, MpiSymbolRateHz);
    MpiMul(&MpiVar0, MpiVar0, MpiConst);
    MpiLeftShift(&MpiVar1, MpiCrystalFreqHz, 19);
    MpiDiv(&MpiVarN1, &MpiNone, MpiVar0, MpiVar1);

    MpiMul(&MpiVar0, MpiTrOffsetN2Int, MpiSymbolRateHz);
    MpiMul(&MpiVar0, MpiVar0, MpiConst);
    MpiLeftShift(&MpiVar1, MpiCrystalFreqHz, 23);
    MpiDiv(&MpiVarN2, &MpiNone, MpiVar0, MpiVar1);

    MpiSub(&MpiVar0, MpiVarN1, MpiVarN2);

    MpiGetValue(MpiVar0, pTrOffsetPpm);


    return FUNCTION_SUCCESS;


error_status_get_demod_registers:
error_status_set_demod_registers:
error_status_get_demod_symbol_rate:
    return FUNCTION_ERROR;
}





/**

@see   ATSC_DEMOD_FP_GET_CR_OFFSET_HZ

*/
int
rtd2885_atsc_GetCrOffsetHz(
    ATSC_DEMOD_MODULE *pDemod,
    long *pCrOffsetHz
    )
{
    unsigned long CrystalFreqHz;

    unsigned long CrOffsetN1Binary;
    long          CrOffsetN1Int;

    unsigned long CrOffsetN2BinaryLsb, CrOffsetN2BinaryMsb;
    long          CrOffsetN2IntMsb;

    MPI MpiCrystalFreqHz, MpiCrOffsetN1Int, MpiCrOffsetN2Int, MpiCrOffsetN2BinaryLsb;
    MPI MpiVarN1, MpiVarN2, MpiVar;



    // Get demod crystal frequency in Hz.
    pDemod->GetCrystalFreqHz(pDemod, &CrystalFreqHz);


    // Set RO_STROBE with 0x1, then set RO_STROBE with 0x0.
    // Note: RO_STROBE is a write-only register.
    if(pDemod->SetRegBitsWithPage(pDemod, ATSC_RO_STROBE, 0x1) != FUNCTION_SUCCESS)
        goto error_status_set_demod_registers;

    if(pDemod->SetRegBitsWithPage(pDemod, ATSC_RO_STROBE, 0x0) != FUNCTION_SUCCESS)
        goto error_status_set_demod_registers;


    // Get CR offset N1 binary value from RO_CR_OFFSET_N1 register bits.
    // Note: The function GetRegBitsWithPage() will set register page automatically.
    if(pDemod->GetRegBitsWithPage(pDemod, ATSC_RO_CR_OFFSET_N1, &CrOffsetN1Binary) != FUNCTION_SUCCESS)
        goto error_status_get_demod_registers;

    // Convert CR offset N1 binary value to signed integer.
    CrOffsetN1Int = BinToSignedInt(CrOffsetN1Binary, RTD2885_ATSC_CR_OFFSET_N1_BIT_NUM);


    // Get CR offset N2 LSB and MSB binary value from RO_CR_OFFSET_N2_25_0 and RO_CR_OFFSET_N2_32_26 register bits.
    // Note: The function GetRegBitsWithPage() will set register page automatically.
    if(pDemod->GetRegBitsWithPage(pDemod, ATSC_RO_CR_OFFSET_N2_25_0, &CrOffsetN2BinaryLsb) != FUNCTION_SUCCESS)
        goto error_status_get_demod_registers;

    if(pDemod->GetRegBitsWithPage(pDemod, ATSC_RO_CR_OFFSET_N2_32_26, &CrOffsetN2BinaryMsb) != FUNCTION_SUCCESS)
        goto error_status_get_demod_registers;

    // Convert CR offset N2 MSB binary value to signed integer.
    // Note: Because CR offset N2 is a 33-bit signed integer, we can not use BinToSignedInt() directly.
    //       Original formula:   CrOffsetN2Int = BinToSignedInt(CrOffsetN2Binary)
    //       Adjusted formula:   CrOffsetN2Int = (BinToSignedInt(CrOffsetN2BinaryMsb) << 26) + CrOffsetN2BinaryLsb
    CrOffsetN2IntMsb = BinToSignedInt(CrOffsetN2BinaryMsb, RTD2885_ATSC_CR_OFFSET_N2_32_26_BIT_NUM);


    // Get CR offset in Hz.
    // Note: Original formula:   CrOffsetHz   = CrOffsetN1Hz + CrOffsetN2Hz
    //                           CrOffsetN1Hz = (CrOffsetN1Int * CrystalFreqHz) / pow(2, 20)
    //                           CrOffsetN2Hz = (CrOffsetN2Int * CrystalFreqHz) / pow(2, 37)
    //       Adjusted formula:   CrOffsetHz   = CrOffsetN1Hz + CrOffsetN2Hz
    //                           CrOffsetN1Hz = (CrOffsetN1Int * CrystalFreqHz) >> 20
    //                           CrOffsetN2Hz = (((CrOffsetN2IntMsb << 26) + CrOffsetN2BinaryLsb) * CrystalFreqHz) >> 37
    //       Because the bit number of CrOffsetN2BinaryLsb is less than 32,
    //       there is no signed bit extension issue when use MpiSetValue().
    MpiSetValue(&MpiCrystalFreqHz, CrystalFreqHz);
    MpiSetValue(&MpiCrOffsetN1Int, CrOffsetN1Int);
    MpiSetValue(&MpiCrOffsetN2Int, CrOffsetN2IntMsb);
    MpiSetValue(&MpiCrOffsetN2BinaryLsb, (long)CrOffsetN2BinaryLsb);

    MpiLeftShift(&MpiCrOffsetN2Int, MpiCrOffsetN2Int, RTD2885_ATSC_CR_OFFSET_N2_25_0_BIT_NUM);
    MpiAdd(&MpiCrOffsetN2Int, MpiCrOffsetN2Int, MpiCrOffsetN2BinaryLsb);

    MpiMul(&MpiVarN1, MpiCrOffsetN1Int, MpiCrystalFreqHz);
    MpiRightShift(&MpiVarN1, MpiVarN1, 20);

    MpiMul(&MpiVarN2, MpiCrOffsetN2Int, MpiCrystalFreqHz);
    MpiRightShift(&MpiVarN2, MpiVarN2, 37);

    MpiAdd(&MpiVar, MpiVarN1, MpiVarN2);

    MpiGetValue(MpiVar, pCrOffsetHz);


    return FUNCTION_SUCCESS;


error_status_get_demod_registers:
error_status_set_demod_registers:
    return FUNCTION_ERROR;
}





/**

@see   ATSC_DEMOD_FP_IS_AAGC_LOCKED

*/
int
rtd2885_atsc_IsAagcLocked(
    ATSC_DEMOD_MODULE *pDemod,
    int *pAnswer
    )
{
    unsigned long RoAagcLock;



    // Set RO_STROBE with 0x1, then set RO_STROBE with 0x0.
    // Note: RO_STROBE is a write-only register.
    if(pDemod->SetRegBitsWithPage(pDemod, ATSC_RO_STROBE, 0x1) != FUNCTION_SUCCESS)
        goto error_status_set_demod_registers;

    if(pDemod->SetRegBitsWithPage(pDemod, ATSC_RO_STROBE, 0x0) != FUNCTION_SUCCESS)
        goto error_status_set_demod_registers;


    // Get RO_AAGC_LOCK.
    // Note: The function GetRegBitsWithPage() will set register page automatically.
    if(pDemod->GetRegBitsWithPage(pDemod, ATSC_RO_AAGC_LOCK, &RoAagcLock) != FUNCTION_SUCCESS)
        goto error_status_get_demod_registers;


    // Determine AAGC lock status according to RO_AAGC_LOCK.
    switch(RoAagcLock)
    {
        default:
        case 0x0:

            *pAnswer = NO;

            break;

        case 0x1:

            *pAnswer = YES;

            break;
    }


    return FUNCTION_SUCCESS;


error_status_get_demod_registers:
error_status_set_demod_registers:
    return FUNCTION_ERROR;
}





/**

@see   ATSC_DEMOD_FP_IS_FRAME_LOCKED

*/
int
rtd2885_atsc_IsFrameLocked(
    ATSC_DEMOD_MODULE *pDemod,
    int *pAnswer
    )
{
    unsigned long RoCurrentState;



    // Set RO_STROBE with 0x1, then set RO_STROBE with 0x0.
    // Note: RO_STROBE is a write-only register.
    if(pDemod->SetRegBitsWithPage(pDemod, ATSC_RO_STROBE, 0x1) != FUNCTION_SUCCESS)
        goto error_status_set_demod_registers;

    if(pDemod->SetRegBitsWithPage(pDemod, ATSC_RO_STROBE, 0x0) != FUNCTION_SUCCESS)
        goto error_status_set_demod_registers;


    // Get RO_CURRENT_STATE.
    // Note: The function GetRegBitsWithPage() will set register page automatically.
    if(pDemod->GetRegBitsWithPage(pDemod, ATSC_RO_CURRENT_STATE, &RoCurrentState) != FUNCTION_SUCCESS)
        goto error_status_get_demod_registers;


    // Determine frame lock status according to RO_CURRENT_STATE.
    switch(RoCurrentState)
    {
        default:

            *pAnswer = NO;

            break;

        case 32:
        case 33:
        case 34:

            *pAnswer = YES;

            break;
    }


    return FUNCTION_SUCCESS;


error_status_get_demod_registers:
error_status_set_demod_registers:
    return FUNCTION_ERROR;
}





/**

@see   ATSC_DEMOD_FP_GET_ERROR_RATE

*/
int
rtd2885_atsc_GetErrorRate(
    ATSC_DEMOD_MODULE *pDemod,
    unsigned long TestVolume,
    unsigned int WaitTimeMsMax,
    unsigned long *pByteErrorRateNum,
    unsigned long *pByteErrorRateDen,
    unsigned long *pPacketErrorRateNum,
    unsigned long *pPacketErrorRateDen
    )
{
    BASE_INTERFACE_MODULE *pBaseInterface;

    unsigned int i;
    unsigned long TestPacketNum;
    unsigned int  WaitCnt;
    int FrameLock;
    unsigned long AoPktCnt;
    unsigned long UncorrectablePacketNum, ErrorByteNum;



    // Get base interface.
    pBaseInterface = pDemod->pBaseInterface;


    // Calculate test packet number and wait counter value.
    TestPacketNum = 0x1 << (TestVolume * 2 + 4);
    WaitCnt = WaitTimeMsMax / RTD2885_ATSC_BER_WAIT_TIME_MS;


    // Set TEST_VOLUME with test volume.
    // Note: The function SetRegBitsWithPage() will set register page automatically.
    if(pDemod->SetRegBitsWithPage(pDemod, ATSC_TEST_VOLUME, TestVolume) != FUNCTION_SUCCESS)
        goto error_status_set_demod_registers;


    // Clear and enable error counter.
    // Note: The function SetRegBitsWithPage() will set register page automatically.
    if(pDemod->SetRegBitsWithPage(pDemod, ATSC_BER_RST, 0x1) != FUNCTION_SUCCESS)
        goto error_status_set_demod_registers;

    if(pDemod->SetRegBitsWithPage(pDemod, ATSC_BER_RST, 0x0) != FUNCTION_SUCCESS)
        goto error_status_set_demod_registers;


    // Check if error test is finished.
    for(i = 0; i < WaitCnt; i++)
    {
        // Check if demod is frame-locked.
        if(pDemod->IsFrameLocked(pDemod, &FrameLock) != FUNCTION_SUCCESS)
            goto error_status_get_demod_registers;

        if(FrameLock == NO)
            goto error_status_frame_lock;


        // Wait a minute.
        // Note: The input unit of WaitMs() is ms.
        pBaseInterface->WaitMs(pBaseInterface, RTD2885_ATSC_BER_WAIT_TIME_MS);


        // Set RO_STROBE with 0x1, then set RO_STROBE with 0x0.
        // Note: RO_STROBE is a write-only register.
        if(pDemod->SetRegBitsWithPage(pDemod, ATSC_RO_STROBE, 0x1) != FUNCTION_SUCCESS)
            goto error_status_set_demod_registers;

        if(pDemod->SetRegBitsWithPage(pDemod, ATSC_RO_STROBE, 0x0) != FUNCTION_SUCCESS)
            goto error_status_set_demod_registers;


        // Check if error test is finished.
        // Note: The function GetRegBitsWithPage() will set register page automatically.
        if(pDemod->GetRegBitsWithPage(pDemod, ATSC_AO_PKT_CNT, &AoPktCnt) != FUNCTION_SUCCESS)
            goto error_status_get_demod_registers;

        if(AoPktCnt == TestPacketNum)
            break;
    }


    // Check time-out status.
    if(i == WaitCnt)
        goto error_status_time_out;


    // Get uncorrectable packet number from AO_UN_CORR_CNT.
    // Note: The function GetRegBitsWithPage() will set register page automatically.
    if(pDemod->GetRegBitsWithPage(pDemod, ATSC_AO_UN_CORR_CNT, &UncorrectablePacketNum) != FUNCTION_SUCCESS)
        goto error_status_get_demod_registers;


    // Get error byte number from AO_ERR_BYTE_CNT.
    // Note: The function GetRegBitsWithPage() will set register page automatically.
    if(pDemod->GetRegBitsWithPage(pDemod, ATSC_AO_ERR_BYTE_CNT, &ErrorByteNum) != FUNCTION_SUCCESS)
        goto error_status_get_demod_registers;


    // Set byte error rate numerator and denominator.
    *pByteErrorRateNum = ErrorByteNum;
    *pByteErrorRateDen = 207 * TestPacketNum;


    // Set packet error rate numerator and denominator.
    *pPacketErrorRateNum = UncorrectablePacketNum;
    *pPacketErrorRateDen = TestPacketNum;


    return FUNCTION_SUCCESS;


error_status_set_demod_registers:
error_status_get_demod_registers:
error_status_frame_lock:
error_status_time_out:
    return FUNCTION_ERROR;
}





/**

@see   ATSC_DEMOD_FP_GET_SNR_DB

*/
int
rtd2885_atsc_GetSnrDb(
    ATSC_DEMOD_MODULE *pDemod,
    long *pSnrDbNum,
    long *pSnrDbDen
    )
{
    unsigned long Mse2Lsb, Mse2Msb;

    MPI MpiMse, MpiMse2Lsb, MpiVar;
    long Var;



    // Set RO_STROBE with 0x1, then set RO_STROBE with 0x0.
    // Note: RO_STROBE is a write-only register.
    if(pDemod->SetRegBitsWithPage(pDemod, ATSC_RO_STROBE, 0x1) != FUNCTION_SUCCESS)
        goto error_status_set_demod_registers;

    if(pDemod->SetRegBitsWithPage(pDemod, ATSC_RO_STROBE, 0x0) != FUNCTION_SUCCESS)
        goto error_status_set_demod_registers;


    // Get MSE2 LSB and Msb value from RO_MSE2_23_0 and RO_MSE2_33_24.
    // Note: The function GetRegBitsWithPage() will set register page automatically.
    if(pDemod->GetRegBitsWithPage(pDemod, ATSC_RO_MSE2_23_0, &Mse2Lsb) != FUNCTION_SUCCESS)
        goto error_status_get_demod_registers;

    if(pDemod->GetRegBitsWithPage(pDemod, ATSC_RO_MSE2_33_24, &Mse2Msb) != FUNCTION_SUCCESS)
        goto error_status_get_demod_registers;



    // SNR dB formula
    // Original formula: SNR_dB = 10 * log10(21 * pow(2, 24) / MSE)
    // Adjusted formula: SNR_dB = (SNR_DB_NUM_CONST - 10 * log2(MSE) * pow(2, SNR_FRAC_BIT_NUM)) / SNR_DB_DEN
    //                   SNR_DB_NUM_CONST = 10 * log2(21 * pow(2, 26)) * pow(2, SNR_FRAC_BIT_NUM)
    //                   SNR_DB_DEN       = log2(10) * pow(2, SNR_FRAC_BIT_NUM)


    // Calculate SNR dB numerator.
    // Note: Mse = (Mse2Msb << 24) + Mse2Lsb
    //       Because the bit numbers of Mse2Msb and Mse2Lsb are less than 32,
    //       there is no signed bit extension issue when use MpiSetValue().
    MpiSetValue(&MpiMse, (long)Mse2Msb);
    MpiSetValue(&MpiMse2Lsb, (long)Mse2Lsb);

    MpiLeftShift(&MpiMse, MpiMse, RTD2885_ATSC_MSE2_23_0_BIT_NUM);
    MpiAdd(&MpiMse, MpiMse, MpiMse2Lsb);

    MpiLog2(&MpiVar, MpiMse, RTD2885_ATSC_SNR_FRAC_BIT_NUM);

    MpiGetValue(MpiVar, &Var);

    *pSnrDbNum = RTD2885_ATSC_SNR_DB_NUM_CONST - 10 * Var;


    // Set SNR dB denominator.
    *pSnrDbDen = RTD2885_ATSC_SNR_DB_DEN;


    return FUNCTION_SUCCESS;


error_status_get_demod_registers:
error_status_set_demod_registers:
    return FUNCTION_ERROR;
}





/**

@see   ATSC_DEMOD_FP_GET_SIGNAL_STRENGTH

*/
int
rtd2885_atsc_GetSignalStrength(
    ATSC_DEMOD_MODULE *pDemod,
    unsigned long *pSignalStrength
    )
{
    int FrameLock;
    long IfAgcValue;



    // Get demod frame lock status.
    if(pDemod->IsFrameLocked(pDemod, &FrameLock) != FUNCTION_SUCCESS)
        goto error_status_get_demod_registers;

    // If demod is not frame-locked, set signal strength with zero.
    if(FrameLock == NO)
    {
        *pSignalStrength = 0;
        goto success_status_non_frame_lock;
    }


    // Get IF AGC value.
    if(pDemod->GetIfAgc(pDemod, &IfAgcValue) != FUNCTION_SUCCESS)
        goto error_status_get_demod_registers;


    // Determine signal strength according to IF AGC value.
    // Note: Map IF AGC value (8191 ~ -8192) to signal strength (0 ~ 100) linearly.
    *pSignalStrength = (819100 - IfAgcValue * 100) / 16383;


success_status_non_frame_lock:
    return FUNCTION_SUCCESS;


error_status_get_demod_registers:
    return FUNCTION_ERROR;
}





/**

@see   ATSC_DEMOD_FP_GET_SIGNAL_QUALITY

*/
int
rtd2885_atsc_GetSignalQuality(
    ATSC_DEMOD_MODULE *pDemod,
    unsigned long *pSignalQuality
    )
{
    int FrameLock;
    long SnrDbNum, SnrDbDen, SnrDbUnit0p01;



    // Get demod frame lock status.
    if(pDemod->IsFrameLocked(pDemod, &FrameLock) != FUNCTION_SUCCESS)
        goto error_status_get_demod_registers;

    // If demod is not frame-locked, set signal quality with zero.
    if(FrameLock == NO)
    {
        *pSignalQuality = 0;
        goto success_status_non_frame_lock;
    }


    // Get SNR in dB from demod.
    if(pDemod->GetSnrDb(pDemod, &SnrDbNum, &SnrDbDen) != FUNCTION_SUCCESS)
        goto error_status_get_demod_registers;


    // Determine signal quality according to SNR value.
    // Note: Map SNR value 14 ~ 20 to signal quality (0 ~ 100) linearly.
    //       If SNR value < 14, signal quality is 0.
    //       If SNR value > 20, signal quality is 100.
    SnrDbUnit0p01 = (SnrDbNum * 100) / SnrDbDen;

    if(SnrDbUnit0p01 < 1400)
    {
        *pSignalQuality = 0;
    }
    else if(SnrDbUnit0p01 > 2000)
    {
        *pSignalQuality = 100;
    }
    else
    {
        *pSignalQuality = (SnrDbUnit0p01 * 100 - 140000) / 600;
    }


success_status_non_frame_lock:
    return FUNCTION_SUCCESS;


error_status_get_demod_registers:
    return FUNCTION_ERROR;
}





/**

@see   ATSC_DEMOD_FP_UPDATE_FUNCTION

*/
int
rtd2885_atsc_UpdateFunction(
    ATSC_DEMOD_MODULE *pDemod
    )
{
    // RTD2885 ATSC does not use UpdateFunction(), so we just return FUNCTION_SUCCESS.
    return FUNCTION_SUCCESS;
}





/**

@see   ATSC_DEMOD_FP_RESET_FUNCTION

*/
int
rtd2885_atsc_ResetFunction(
    ATSC_DEMOD_MODULE *pDemod
    )
{
    // RTD2885 ATSC does not use UpdateFunction(), so we just return FUNCTION_SUCCESS.
    return FUNCTION_SUCCESS;
}





/**

@see   I2C_BRIDGE_FP_FORWARD_I2C_READING_CMD

*/
int
rtd2885_atsc_ForwardI2cReadingCmd(
    I2C_BRIDGE_MODULE *pI2cBridge,
    unsigned char DeviceAddr,
    unsigned char *pReadingBytes,
    unsigned long ByteNum
    )
{
    ATSC_DEMOD_MODULE *pDemod;
    BASE_INTERFACE_MODULE *pBaseInterface;



    // Get demod module and tuner device address.
    pDemod = (ATSC_DEMOD_MODULE *)pI2cBridge->pPrivateData;

    
    // Get base interface.
    pBaseInterface = pDemod->pBaseInterface;


    // Enable demod I2C relay.
    if(pDemod->SetRegBitsWithPage(pDemod, ATSC_OPT_I2C_RELAY, 0x1) != FUNCTION_SUCCESS)
        goto error_status_set_demod_registers;


    // Send I2C reading command.
    if(pBaseInterface->I2cRead(pBaseInterface, DeviceAddr, pReadingBytes, ByteNum) != FUNCTION_SUCCESS)
        goto error_send_i2c_reading_command;


    return FUNCTION_SUCCESS;


error_send_i2c_reading_command:
error_status_set_demod_registers:
    return FUNCTION_ERROR;
}





/**

@see   I2C_BRIDGE_FP_FORWARD_I2C_WRITING_CMD

*/
int
rtd2885_atsc_ForwardI2cWritingCmd(
    I2C_BRIDGE_MODULE *pI2cBridge,
    unsigned char DeviceAddr,
    const unsigned char *pWritingBytes,
    unsigned long ByteNum
    )
{
    ATSC_DEMOD_MODULE *pDemod;
    BASE_INTERFACE_MODULE *pBaseInterface;



    // Get demod module and tuner device address.
    pDemod = (ATSC_DEMOD_MODULE *)pI2cBridge->pPrivateData;

    
    // Get base interface.
    pBaseInterface = pDemod->pBaseInterface;


    // Enable demod I2C relay.
    if(pDemod->SetRegBitsWithPage(pDemod, ATSC_OPT_I2C_RELAY, 0x1) != FUNCTION_SUCCESS)
        goto error_status_set_demod_registers;


    // Send I2C writing command.
    if(pBaseInterface->I2cWrite(pBaseInterface, DeviceAddr, pWritingBytes, ByteNum) != FUNCTION_SUCCESS)
        goto error_send_i2c_writing_command;


    return FUNCTION_SUCCESS;


error_send_i2c_writing_command:
error_status_set_demod_registers:
    return FUNCTION_ERROR;
}





/**

@brief   Initialize register table

RTD2885 ATSC builder will use rtd2885_atsc_InitRegTable() to initialize register table.


@param [in]   pDemod   The demod module pointer


@see   BuildRtd2885AtscModule()

*/
void
rtd2885_atsc_InitRegTable(
    ATSC_DEMOD_MODULE *pDemod
    )
{
    static const ATSC_PRIMARY_REG_ENTRY PrimaryRegTable[RTD2885_ATSC_REG_TABLE_LEN] =
    {
        // Generality
        // RegBitName,                      PageNo,     RegStartAddr,   Msb,    Lsb
        {ATSC_SYS_VERSION,                  0x0,        0x1,            7,      0   },
        {ATSC_OPT_I2C_RELAY,                0x0,        0x6,            7,      7   },
        {ATSC_OPT_SOFT_RESET_ATSC,          0x0,        0x5,            1,      1   },

        // Miscellany
        // RegBitName,                      PageNo,     RegStartAddr,   Msb,    Lsb
        {ATSC_RO_STROBE,                    0x2f,       0x1,            0,      0   },

        // AAGC
        // RegBitName,                      PageNo,     RegStartAddr,   Msb,    Lsb
        {ATSC_OPT_RF_AAGC_DRIVE,            0x0,        0x8,            1,      1   },
        {ATSC_OPT_PAR_RF_SD_IB,             0x0,        0x8,            2,      2   },
        {ATSC_OPT_RF_AAGC_OE,               0x0,        0x8,            3,      3   },
        {ATSC_OPT_RF_AGC_DRIVING,           0x0,        0x16,           3,      3   },
        {ATSC_RF_AGC_MAX,                   0x21,       0xd,            7,      0   },
        {ATSC_RF_AGC_MIN,                   0x21,       0xc,            7,      0   },
        {ATSC_OPT_IF_AAGC_DRIVE,            0x0,        0x8,            5,      5   },
        {ATSC_OPT_PAR_IF_SD_IB,             0x0,        0x8,            6,      6   },
        {ATSC_OPT_IF_AAGC_OE,               0x0,        0x8,            7,      7   },
        {ATSC_OPT_IF_AGC_DRIVING,           0x0,        0x16,           2,      2   },
        {ATSC_IF_AGC_MAX,                   0x21,       0xb,            7,      0   },
        {ATSC_IF_AGC_MIN,                   0x21,       0xa,            7,      0   },
        {ATSC_AAGC_TARGET,                  0x21,       0x3,            7,      0   },
        {ATSC_AGC_MODE,                     0x21,       0x4,            5,      5   },
        {ATSC_VTOP,                         0x21,       0x8,            6,      0   },
        {ATSC_KRF,                          0x21,       0x9,            5,      0   },
        {ATSC_RO_AAGC_LOCK,                 0x2f,       0xa,            7,      7   },
        {ATSC_RO_RF_AAGC,                   0x2f,       0x2,            13,     0   },
        {ATSC_RO_IF_AAGC,                   0x2f,       0x4,            13,     0   },

        // DAGC
        // RegBitName,                      PageNo,     RegStartAddr,   Msb,    Lsb
        {ATSC_RO_DAGC_LEVEL_Q,              0x2f,       0xe,            12,     0   },

        // Demod status
        // RegBitName,                      PageNo,     RegStartAddr,   Msb,    Lsb
        {ATSC_RO_CURRENT_STATE,             0x2f,       0x9,            7,      2   },

        // DDC
        // RegBitName,                      PageNo,     RegStartAddr,   Msb,    Lsb
        {ATSC_DDC_PHASE_INC,                0x21,       0xf,            19,     0   },
        {ATSC_SPEC_INV_ON,                  0x21,       0x11,           4,      4   },
        {ATSC_CR_CONST_INC,                 0x26,       0x5a,           16,     0   },
        {ATSC_TR_PHASE_INC,                 0x21,       0x65,           26,     0   },

        // Carrier recovery
        // RegBitName,                      PageNo,     RegStartAddr,   Msb,    Lsb
        {ATSC_RO_CR_OFFSET_N1,              0x2f,       0x80,           17,     0   },
        {ATSC_RO_CR_OFFSET_N2_25_0,         0x2f,       0x82,           31,     6   },
        {ATSC_RO_CR_OFFSET_N2_32_26,        0x2f,       0x86,           6,      0   },

        // Timing recovery
        // RegBitName,                      PageNo,     RegStartAddr,   Msb,    Lsb
        {ATSC_RO_TR_OFFSET_N1,              0x2f,       0x99,           10,     0   },
        {ATSC_RO_TR_OFFSET_N2,              0x2f,       0x12,           15,     0   },

        // Equalizer
        // RegBitName,                      PageNo,     RegStartAddr,   Msb,    Lsb
        {ATSC_RO_MSE2_23_0,                 0x2f,       0x24,           23,     0   },
        {ATSC_RO_MSE2_33_24,                0x2f,       0x27,           9,      0   },

        // Outer
        // RegBitName,                      PageNo,     RegStartAddr,   Msb,    Lsb
        {ATSC_BER_RST,                      0x28,       0x56,           7,      7   },
        {ATSC_BER_HOLD,                     0x28,       0x56,           6,      6   },
        {ATSC_DIS_AUTO_MODE,                0x28,       0x56,           5,      5   },
        {ATSC_TEST_VOLUME,                  0x28,       0x56,           2,      0   },
        {ATSC_AO_ERR_BYTE_CNT,              0x2f,       0xe6,           27,     0   },
        {ATSC_AO_PKT_CNT,                   0x2f,       0xea,           23,     0   },
        {ATSC_AO_CORR_CNT,                  0x2f,       0xed,           23,     0   },
        {ATSC_AO_UN_CORR_CNT,               0x2f,       0xf0,           23,     0   },

        // MPEG TS output interface
        // RegBitName,                      PageNo,     RegStartAddr,   Msb,    Lsb
        {ATSC_OPT_M_OEN,                    0x0,        0x6,            6,      6   },
        {ATSC_FIX_TEI,                      0x28,       0x52,           0,      0   },
        {ATSC_CKOUTPAR,                     0x28,       0x52,           1,      1   },
        {ATSC_CKOUT_PWR,                    0x28,       0x52,           2,      2   },
        {ATSC_SYNC_DUR,                     0x28,       0x52,           3,      3   },
        {ATSC_ERR_DUR,                      0x28,       0x52,           4,      4   },
        {ATSC_ERR_LVL,                      0x28,       0x52,           5,      5   },
        {ATSC_VAL_LVL,                      0x28,       0x52,           6,      6   },
        {ATSC_SERIAL,                       0x28,       0x53,           0,      0   },
        {ATSC_SER_LSB,                      0x28,       0x53,           1,      1   },
        {ATSC_CDIV_PH0,                     0x28,       0x54,           3,      0   },
        {ATSC_CDIV_PH1,                     0x28,       0x54,           7,      4   },

        // Smart antenna
        // RegBitName,                      PageNo,     RegStartAddr,   Msb,    Lsb
        {ATSC_OPT_ANT_IO_MODE,              0x0,        0x15,           2,      0   },
        {ATSC_OPT_ANT_PAD_OE,               0x0,        0x15,           3,      3   },
        {ATSC_OPT_ANT_DRIVING,              0x0,        0x15,           4,      4   },
        {ATSC_OPT_ANT_RESET_N,              0x22,       0x4,            0,      0   },
        {ATSC_OPT_ANT_ENABLE,               0x22,       0x4,            1,      1   },
    };


    int i;
    int RegBitName;



    // Initialize register table according to primary register table.
    // Note: 1. Register table rows are sorted by register bit name key.
    //       2. The default value of the IsAvailable variable is "NO".
    for(i = 0; i < ATSC_REG_TABLE_LEN_MAX; i++)
        pDemod->RegTable[i].IsAvailable  = NO;

    for(i = 0; i < RTD2885_ATSC_REG_TABLE_LEN; i++)
    {
        RegBitName = PrimaryRegTable[i].RegBitName;

        pDemod->RegTable[RegBitName].IsAvailable  = YES;
        pDemod->RegTable[RegBitName].PageNo       = PrimaryRegTable[i].PageNo;
        pDemod->RegTable[RegBitName].RegStartAddr = PrimaryRegTable[i].RegStartAddr;
        pDemod->RegTable[RegBitName].Msb          = PrimaryRegTable[i].Msb;
        pDemod->RegTable[RegBitName].Lsb          = PrimaryRegTable[i].Lsb;
    }


    return;
}





/**

@brief   Set I2C bridge module demod arguments.

RTD2885 ATSC builder will use rtd2885_atsc_BuildI2cBridgeModule() to set I2C bridge module demod arguments.


@param [in]   pDemod   The demod module pointer


@see   BuildRtd2885AtscModule()

*/
void
rtd2885_atsc_BuildI2cBridgeModule(
    ATSC_DEMOD_MODULE *pDemod
    )
{
    I2C_BRIDGE_MODULE *pI2cBridge;



    // Get I2C bridge module.
    pI2cBridge = pDemod->pI2cBridge;

    // Set I2C bridge module demod arguments.
    pI2cBridge->pPrivateData = (void *)pDemod;
    pI2cBridge->ForwardI2cReadingCmd = rtd2885_atsc_ForwardI2cReadingCmd;
    pI2cBridge->ForwardI2cWritingCmd = rtd2885_atsc_ForwardI2cWritingCmd;


    return;
}












