/*
 *  Realtek rtl2810b ATSC demodulator driver
 *
 *  Copyright (C) 2011 Liuyi <Bill.Liu@conexant.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/string.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include "dvb_frontend.h"
#include "rtl2810b-core.h"
#include "demod_rtd2885_atsc.h"

#define RTL2810B_ATSC_TDA18272_INIT_REG_TABLE_LEN        13

struct rtl2810b_state {

	struct i2c_adapter *i2c;

	/* configuration settings */
	const struct rtl2810b_config *config;

	struct dvb_frontend frontend;

	fe_modulation_t current_modulation;
	unsigned int first_tune:1;

	u32 current_frequency;
	int if_freq;

	u8 inversion;

	BASE_INTERFACE_MODULE *pBaseInterface;
	BASE_INTERFACE_MODULE BaseInterfaceModuleMemory;

	ATSC_DEMOD_MODULE *pDemod;
	ATSC_DEMOD_MODULE AtscDemodModuleMemory;

	I2C_BRIDGE_MODULE I2cBridgeModuleMemory;

};

static int debug;

#define dprintk(arg...) do {	\
	if (debug)		\
		printk(arg);	\
	} while (0)

#define RTL2832_TDA18272_ADDITIONAL_INIT_REG_TABLE_LEN		21

static int CustomI2cRead(
	BASE_INTERFACE_MODULE *pBaseInterface,
	unsigned char DeviceAddr,
	unsigned char *pReadingBytes,
	unsigned long ByteNum
	)
{
	printk(KERN_INFO " Enter CustomI2cRead\n");
	return FUNCTION_ERROR;
}

static int CustomI2cWriteThenRead(
	BASE_INTERFACE_MODULE *pBaseInterface,
	unsigned char DeviceAddr,
	const unsigned char *pWritingBytes,
	unsigned long ByteWNum,
	unsigned char *pReadingBytes,
	unsigned long ByteRNum
	)
{
	struct rtl2810b_state *state = NULL;

	unsigned char buf = 0x00;
	unsigned char buf8[8];
	int ret = 0;
	unsigned char i = 0;

	struct i2c_msg msgwr[] = {
		{ .addr = DeviceAddr, .flags = 0,
		  .buf = &buf, .len = 1 },
		{ .addr = DeviceAddr, .flags = I2C_M_RD,
		  .buf = buf8, .len = 1 }
	};

	pBaseInterface->GetUserDefinedDataPointer(pBaseInterface,
		 (void **)&state);

	for (i = 0; i < ByteRNum; i++) {
		buf = pWritingBytes[0]+i;
		ret = i2c_transfer(state->i2c, msgwr, 2);
		pReadingBytes[i] = buf8[0];
	}

	if (ret != 2)
		printk(KERN_INFO "ERROR: i2c_transfer returned: %d\n", ret);

   return 0;
}


static int CustomI2cWrite(
	BASE_INTERFACE_MODULE *pBaseInterface,
	unsigned char DeviceAddr,
	const unsigned char *pWritingBytes,
	unsigned long ByteNum
)
{
	struct rtl2810b_state *state = NULL;
	int ret = 0;
	u8 buf[4];
	u8 i = 0;
	struct i2c_msg msg = {
		.addr = DeviceAddr, .flags = 0, .buf = buf, .len = 2 };

	pBaseInterface->GetUserDefinedDataPointer(pBaseInterface ,
		 (void **)&state);

	for (i = 0; i < ByteNum - 1; i++) {
		buf[0] = pWritingBytes[0]+i;
		buf[1] = pWritingBytes[i+1];
		ret = i2c_transfer(state->i2c, &msg, 1);
	}

	return 0;
}

static void CustomWaitMs(
	BASE_INTERFACE_MODULE *pBaseInterface,
	unsigned long WaitTimeMs
	)
{

}

static int rtl2810b_sleep(struct dvb_frontend *fe)
{
	return 0;
}

/* Talk to the demod, set the FEC, GUARD, QAM settings etc */
static int rtl2810b_set_frontend(struct dvb_frontend *fe,
				struct dvb_frontend_parameters *p)
{
	struct rtl2810b_state *state = fe->demodulator_priv;
	ATSC_DEMOD_MODULE *pDemod = state->pDemod;

	if (p->frequency == state->current_frequency) {
		/*current_frequency = p->frequency; */
		/*state->current_frequency = p->frequency; */
	} else {
		printk(KERN_INFO " setting tuner...p->u.vsb.modulation=0x%x\n",
			p->u.vsb.modulation);
		fe->ops.tuner_ops.set_params(fe, p);
		msleep(300);

		pDemod->SetIfFreqHz(pDemod, 3250000);

		/*fe->ops.tuner_ops.set_params(fe,p); */
		/*Soft Reset chip*/
		msleep(30);
		pDemod->ResetFunction(pDemod);
		pDemod->SoftwareReset(pDemod);

	}

	state->current_frequency = p->frequency;

	return 0;
}
static int rtl2810b_init_demod(struct dvb_frontend *fe)
{
	struct rtl2810b_state *state = fe->demodulator_priv;
	unsigned long IfFreqHz;
	int SpectrumMode;

	ATSC_DEMOD_MODULE *pDemod = state->pDemod;

	typedef struct {
		int RegBitName;
		unsigned long Value;
	}
	REG_VALUE_ENTRY;

	int i;
	int RegBitName;
	unsigned long Value;


	static const REG_VALUE_ENTRY
AdditionalInitRegValueTable[RTL2810B_ATSC_TDA18272_INIT_REG_TABLE_LEN] = {
	/* RegBitName,                  Value*/
	{ATSC_OPT_RF_AAGC_DRIVE,        0x0     },
	{ATSC_OPT_IF_AAGC_DRIVE,        0x0     },
	{ATSC_OPT_PAR_RF_SD_IB,         0x1     },
	{ATSC_OPT_PAR_IF_SD_IB,         0x1     },
	{ATSC_OPT_RF_AGC_DRIVING,       0x0     },
	{ATSC_OPT_IF_AGC_DRIVING,       0x0     },
	{ATSC_OPT_RF_AAGC_OE,           0x0     },
	{ATSC_OPT_IF_AAGC_OE,           0x1     },
	{ATSC_RF_AGC_MAX,               0x0     },
	{ATSC_RF_AGC_MIN,               0x0     },
	{ATSC_IF_AGC_MAX,               0x7f    },
	{ATSC_IF_AGC_MIN,               0x80    },
	{ATSC_AGC_MODE,                 0x0     },
	};

	/* test register read write*/
	unsigned long ReadingValue = 10;

	IfFreqHz      = IF_FREQ_3250000HZ;
	SpectrumMode  = SPECTRUM_NORMAL;

	/* ==== Initialize DTMB demod and set its parameters =====*/

	/* Initialize demod.*/
	pDemod->Initialize(pDemod);



	pDemod->SetIfFreqHz(pDemod,      IfFreqHz); /* 20100825 need to check*/

	pDemod->SetSpectrumMode(pDemod,  SpectrumMode);

	/* Set demod registers.*/
	for (i = 0; i < RTL2810B_ATSC_TDA18272_INIT_REG_TABLE_LEN;
		 i++) {
		/* Get register bit name and its value.*/
		RegBitName = AdditionalInitRegValueTable[i].RegBitName;
		Value      = AdditionalInitRegValueTable[i].Value;

		/* Set demod registers*/
		if (pDemod->SetRegBitsWithPage(pDemod, RegBitName, Value) !=
			 FUNCTION_SUCCESS) {
			break;
		}
	}

	if (pDemod->SetRegPage(pDemod, 0) != FUNCTION_SUCCESS)
		printk(KERN_INFO " ##### ##### SetRegPage fail \n");


	if (pDemod->GetRegMaskBits(pDemod, 1, 7, 0, &ReadingValue) !=
		 FUNCTION_SUCCESS)
		printk(KERN_INFO " ##### ##### GetRegMaskBits fail \n");

	printk(KERN_INFO " read register page=0x01 addr=0x%x readvalue = 0x%x",
		1, (int)ReadingValue);

	if (pDemod->SetRegPage(pDemod, 0x28) != FUNCTION_SUCCESS)
		printk(KERN_INFO " ##### ##### SetRegPage fail 0x28 \n");


	if (pDemod->GetRegMaskBits(pDemod, 0x52, 1, 1, &ReadingValue) !=
		 FUNCTION_SUCCESS)
		printk(KERN_INFO " ##### ##### SetRegMaskBits fail 0x52\n");


	printk(KERN_INFO " read register page=0x28 addr=0x52 readvalue = 0x%x",
		(int)ReadingValue);

	if (pDemod->SetRegPage(pDemod, 0x28) != FUNCTION_SUCCESS)
		printk(KERN_INFO " ##### ##### SetRegPage fail 0x28 \n");


	if (pDemod->SetRegMaskBits(pDemod, 0x52, 3, 3, 1) != FUNCTION_SUCCESS)
		printk(KERN_INFO " ##### ##### SetRegMaskBits fail 0x52\n");

	/*DbgPrint(" ##### ##### read register page=0x01 addr=0x%x
	 readvalue = %x ##### ##### ",1,(int)ReadingValue);*/
	if (pDemod->SetRegPage(pDemod, 0x28) != FUNCTION_SUCCESS)
		printk(KERN_INFO " ##### ##### SetRegPage fail 0x28 \n");


	if (pDemod->GetRegMaskBits(pDemod, 0x54, 7, 0, &ReadingValue) !=
		 FUNCTION_SUCCESS)
		printk(KERN_INFO " ##### ##### SetRegMaskBits fail 0x54\n");

	printk(KERN_INFO "read register page=0x28 addr=0x54 readvalue = 0x%x ",
		(int)ReadingValue);

	if (pDemod->SetRegMaskBits(pDemod, 0x54, 7, 0, 0x21) !=
		 FUNCTION_SUCCESS)
		printk(KERN_INFO " ##### ##### SetRegMaskBits fail 0x54\n");


	if (pDemod->SetRegPage(pDemod, 0x28) != FUNCTION_SUCCESS)
		printk(KERN_INFO " ##### ##### SetRegPage fail 0x28 \n");


	if (pDemod->GetRegMaskBits(pDemod, 0x53, 1, 1, &ReadingValue) !=
		 FUNCTION_SUCCESS)
		printk(KERN_INFO " ##### ##### SetRegMaskBits fail 0x53\n");

	printk(KERN_INFO " read register page=0x28 addr=0x53 readvalue = 0x%x ",
		(int)ReadingValue);

	if (pDemod->SetRegMaskBits(pDemod, 0x53, 1, 1, 1) != FUNCTION_SUCCESS)
		printk(KERN_INFO " ##### ##### SetRegMaskBits fail 0x53\n");


	if (pDemod->SetRegPage(pDemod, 0x28) != FUNCTION_SUCCESS)
		printk(KERN_INFO " ##### ##### SetRegPage fail 0x28 \n");

	if (pDemod->GetRegMaskBits(pDemod, 0x53, 1, 1, &ReadingValue) !=
		 FUNCTION_SUCCESS)
		printk(KERN_INFO " ##### ##### SetRegMaskBits fail 0x53\n");


	printk(KERN_INFO "read register page=0x28 addr=0x53 readvalue = 0x%x",
		(int)ReadingValue);

	return FUNCTION_SUCCESS;

}

static int rtl2810b_initial(struct dvb_frontend *fe)
{
	struct rtl2810b_state *state = fe->demodulator_priv;
	unsigned char demod_i2c_address0 = 0x12;
	state->current_frequency = 0;
	printk(KERN_INFO " rtl2810b_initial().\n");

	/* Build base interface module.*/
	BuildBaseInterface(
	&(state->pBaseInterface),
	&(state->BaseInterfaceModuleMemory),
	9,                     /* Set maximum I2C reading byte number with 9.*/
	8,                     /* Set maximum I2C writing byte number with 8.*/
	CustomI2cRead,         /* CustomI2cRead() as basic I2C reading .*/
	CustomI2cWriteThenRead,/* CustomI2cRead() as basic I2C reading .*/
	CustomI2cWrite,        /* CustomI2cWrite() as basic I2C writing .*/
	CustomWaitMs           /* CustomWaitMs() as basic waiting function.*/
	);

	state->pBaseInterface->SetUserDefinedDataPointer(state->pBaseInterface,
				 state);

	/* Build DTMB demod  module.*/
	BuildRtd2885AtscModule(
	&(state->pDemod),
	&(state->AtscDemodModuleMemory),
	&(state->BaseInterfaceModuleMemory),
	&(state->I2cBridgeModuleMemory),
	demod_i2c_address0,        /* Demod I2C address is 0x3e in 8-bit*/
	CRYSTAL_FREQ_27000000HZ,   /* Demod crystal frequency is 27.0 MHz.*/
	TS_INTERFACE_SERIAL        /* Demod TS interface mode is serial.*/

	);
	rtl2810b_init_demod(fe);
	return 0;
}

static int rtl2810b_init(struct dvb_frontend *fe)
{
	struct rtl2810b_state *state = fe->demodulator_priv;
	state->current_frequency = 0;
	printk(KERN_INFO " rtl2810b_init().\n");

	rtl2810b_init_demod(fe);

	return 0;
}

static int rtl2810b_read_status(struct dvb_frontend *fe, fe_status_t *status)
{
	struct rtl2810b_state *state = fe->demodulator_priv;
	ATSC_DEMOD_MODULE *pDemod = state->pDemod;
	int answer;

	pDemod->IsFrameLocked(pDemod, &answer);
	if (answer)
		*status =  FE_HAS_LOCK | FE_HAS_SYNC
		 | FE_HAS_CARRIER | FE_HAS_SIGNAL;
	else
		*status = answer;

	return 1;
}

static int rtl2810b_read_signal_strength(struct dvb_frontend *fe,
					u16 *signal_strength)
{
	struct rtl2810b_state *state = fe->demodulator_priv;
	ATSC_DEMOD_MODULE *pDemod = state->pDemod;
	unsigned long SignalStrength;

	pDemod->GetSignalStrength(pDemod, &SignalStrength);
	*signal_strength = SignalStrength;

	return 0;
}

static int rtl2810b_read_snr(struct dvb_frontend *fe, u16 *snr)
{
	struct rtl2810b_state *state = fe->demodulator_priv;
	ATSC_DEMOD_MODULE *pDemod = state->pDemod;
	long SnrDbNum, SnrDbDen;

	pDemod->GetSnrDb(pDemod, &SnrDbNum, &SnrDbDen);
	*snr = SnrDbNum / SnrDbDen;

	return 0;
}

static int rtl2810b_read_ucblocks(struct dvb_frontend *fe, u32 *ucblocks)
{
	struct rtl2810b_state *state = fe->demodulator_priv;
	ATSC_DEMOD_MODULE *pDemod = state->pDemod;
	int answer;

	pDemod->IsFrameLocked(pDemod, &answer);
	*ucblocks = answer;

	return 0;
}

static int rtl2810b_read_ber(struct dvb_frontend *fe, u32 *ber)
{
	return 0;
}

static int rtl2810b_get_frontend(struct dvb_frontend *fe,
				struct dvb_frontend_parameters *p)
{
	struct rtl2810b_state *state = fe->demodulator_priv;

	p->frequency = state->current_frequency;
	p->u.vsb.modulation = state->current_modulation;

	return 0;
}

static int rtl2810b_get_tune_settings(struct dvb_frontend *fe,
				     struct dvb_frontend_tune_settings *tune)
{
	return 0;
}

static void rtl2810b_release(struct dvb_frontend *fe)
{
	struct rtl2810b_state *state = fe->demodulator_priv;
	kfree(state);
}

static struct dvb_frontend_ops rtl2810b_ops;

struct dvb_frontend *rtl2810b_attach(const struct rtl2810b_config *config,
				    struct i2c_adapter *i2c)
{
	struct rtl2810b_state *state = NULL;

	printk(KERN_INFO " Enter rtl2810b_attach(). attach success!\n");
	/* allocate memory for the internal state */
	state = kmalloc(sizeof(struct rtl2810b_state), GFP_KERNEL);
	if (state == NULL)
		goto error;

	/* setup the state */
	state->config = config;
	state->i2c = i2c;
	state->current_modulation = VSB_8;
	state->inversion = state->config->inversion;

	/* create dvb_frontend */
	memcpy(&state->frontend.ops, &rtl2810b_ops,
	       sizeof(struct dvb_frontend_ops));

	state->frontend.demodulator_priv = state;

	rtl2810b_initial(&state->frontend);

	return &state->frontend;

error:
	kfree(state);
	return NULL;
}
EXPORT_SYMBOL(rtl2810b_attach);

static struct dvb_frontend_ops rtl2810b_ops = {

	.info = {
		 .name = "Realtek rtl2810b ATSC Frontend",
		 .type = FE_ATSC,
		 .frequency_min = 54000000,
		 .frequency_max = 858000000,
		 .frequency_stepsize = 62500,
		 .caps = FE_CAN_QAM_64 | FE_CAN_QAM_256 | FE_CAN_8VSB},

	.init = rtl2810b_init,
	.sleep = rtl2810b_sleep,
	.set_frontend = rtl2810b_set_frontend,
	.get_frontend = rtl2810b_get_frontend,
	.get_tune_settings = rtl2810b_get_tune_settings,
	.read_status = rtl2810b_read_status,
	.read_ber = rtl2810b_read_ber,
	.read_signal_strength = rtl2810b_read_signal_strength,
	.read_snr = rtl2810b_read_snr,
	.read_ucblocks = rtl2810b_read_ucblocks,
	.release = rtl2810b_release,
};

module_param(debug, int, 0644);
MODULE_PARM_DESC(debug, "Enable verbose debug messages");

MODULE_DESCRIPTION("Realtek rtl2810b ATSC Demodulator driver");
MODULE_AUTHOR("Bill Liu");
MODULE_LICENSE("GPL");
