/*
 *  Realtek rtl2832 DVB-T demodulator driver
 *
 *  Copyright (C) 2011 Liuyi <Bill.Liu@conexant.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/string.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include "dvb_frontend.h"
#include "rtl2832-core.h"
#include "demod_rtl2832.h"
struct rtl2832_state {

	struct i2c_adapter *i2c;

	/* configuration settings */
	const struct rtl2832_config *config;

	struct dvb_frontend frontend;

	fe_modulation_t current_modulation;
	unsigned int first_tune:1;

	u32 current_frequency;
	int if_freq;

	u8 inversion;

	BASE_INTERFACE_MODULE *pBaseInterface;
	BASE_INTERFACE_MODULE BaseInterfaceModuleMemory;

	DVBT_DEMOD_MODULE *pDemod;
	DVBT_DEMOD_MODULE DtmbDemodModuleMemory;

	I2C_BRIDGE_MODULE I2cBridgeModuleMemory;

};

static int debug;

#define dprintk(arg...) do {	\
	if (debug)		\
		printk(arg);	\
	} while (0)

#define RTL2832_TDA18272_ADDITIONAL_INIT_REG_TABLE_LEN		21

static int CustomI2cRead(
		BASE_INTERFACE_MODULE *pBaseInterface,
		unsigned char DeviceAddr,
		unsigned char *pReadingBytes,
		unsigned long ByteNum
		)
{
	printk(KERN_INFO " Enter CustomI2cRead\n");
	return FUNCTION_ERROR;
}

static int CustomI2cWriteThenRead(
		BASE_INTERFACE_MODULE *pBaseInterface,
		unsigned char DeviceAddr,
		const unsigned char *pWritingBytes,
		unsigned long ByteWNum,
		unsigned char *pReadingBytes,
		unsigned long ByteRNum
)
{
	struct rtl2832_state *state = NULL;

	unsigned char buf = 0x00;
	unsigned char buf8[8];
	int ret = 0;
	unsigned char i = 0;

	struct i2c_msg msgwr[] = {
		{ .addr = DeviceAddr, .flags = 0,
		  .buf = &buf, .len = 1 },
		{ .addr = DeviceAddr, .flags = I2C_M_RD,
		  .buf = buf8, .len = 1 }
	};


	pBaseInterface->GetUserDefinedDataPointer(pBaseInterface ,
			 (void **)&state);

	for (i = 0; i < ByteRNum; i++) {
		buf = pWritingBytes[0]+i;
		ret = i2c_transfer(state->i2c, msgwr, 2);
		pReadingBytes[i] = buf8[0];
	}

	if (ret != 2)
		printk(KERN_INFO "ERROR: i2c_transfer returned: %d\n", ret);

   return 0;
}


static int CustomI2cWrite(
		BASE_INTERFACE_MODULE *pBaseInterface,
		unsigned char DeviceAddr,
		const unsigned char *pWritingBytes,
		unsigned long ByteNum
		)
{

	struct rtl2832_state *state = NULL;
	int ret = 0;
	u8 buf[4];
	u8 i = 0;
	struct i2c_msg msg = {
		.addr = DeviceAddr, .flags = 0, .buf = buf, .len = 2 };

	pBaseInterface->GetUserDefinedDataPointer(pBaseInterface ,
				 (void **)&state);

	for (i = 0; i < ByteNum - 1; i++) {
		buf[0] = pWritingBytes[0]+i;
		buf[1] = pWritingBytes[i+1];
		ret = i2c_transfer(state->i2c, &msg, 1);
	}

	return 0;
}



static void CustomWaitMs(
	BASE_INTERFACE_MODULE *pBaseInterface,
	unsigned long WaitTimeMs
	)
{

}

static int rtl2832_sleep(struct dvb_frontend *fe)
{
	return 0;
}

/* Talk to the demod, set the FEC, GUARD, QAM settings etc */
static int rtl2832_set_frontend(struct dvb_frontend *fe,
				struct dvb_frontend_parameters *p)
{
	u32 dvb_bandwidth = 2;
	struct rtl2832_state *state = fe->demodulator_priv;
	DVBT_DEMOD_MODULE *pDemod = state->pDemod;

	if (p->frequency == state->current_frequency) {
		/*current_frequency = p->frequency; */
		/*state->current_frequency = p->frequency; */
	} else {
		printk(KERN_INFO " setting tuner...p->u.ofdm.bandwidth=%d\n",
		p->u.ofdm.bandwidth);
		fe->ops.tuner_ops.set_params(fe, p);
		msleep(300);
		/*rtl2832_set_channel_bandwidth(fe, dvb_bandwidth);*/
		pDemod->SetBandwidthMode(pDemod, dvb_bandwidth);
		switch (p->u.ofdm.bandwidth) {
		case BANDWIDTH_6_MHZ:
			dvb_bandwidth = 0;
			pDemod->SetIfFreqHz(pDemod, IF_FREQ_3point25_MHZ);
			break;
		case BANDWIDTH_7_MHZ:
			dvb_bandwidth = 1;
			pDemod->SetIfFreqHz(pDemod, IF_FREQ_3point5_MHZ);
			break;
		case BANDWIDTH_8_MHZ:
			dvb_bandwidth = 2;
			pDemod->SetIfFreqHz(pDemod, IF_FREQ_4_MHZ);
			break;
		default:
			return 0;
		}

		/*rtl2832_set_channel_bandwidth(fe, dvb_bandwidth);*/
		pDemod->SetBandwidthMode(pDemod, dvb_bandwidth);
		switch (p->u.ofdm.bandwidth) {
		case BANDWIDTH_6_MHZ:
			dvb_bandwidth = 0;
			pDemod->SetIfFreqHz(pDemod, IF_FREQ_3point25_MHZ);
			break;
		case BANDWIDTH_7_MHZ:
			dvb_bandwidth = 1;
			pDemod->SetIfFreqHz(pDemod, IF_FREQ_3point5_MHZ);
			break;
		case BANDWIDTH_8_MHZ:
			dvb_bandwidth = 2;
			pDemod->SetIfFreqHz(pDemod, IF_FREQ_4_MHZ);
			break;
		default:
			return 0;
		}
		/*fe->ops.tuner_ops.set_params(fe,p); */
		/*Soft Reset chip*/
		msleep(30);
		pDemod->ResetFunction(pDemod);
		pDemod->SoftwareReset(pDemod);


	}

	state->current_frequency = p->frequency;

	return 0;
}
static int rtl2832_init_demod(struct dvb_frontend *fe)
{
	struct rtl2832_state *state = fe->demodulator_priv;
	unsigned long IfFreqHz;
	int SpectrumMode;

	DVBT_DEMOD_MODULE *pDemod = state->pDemod;
	typedef struct {
		int RegBitName;
		unsigned long Value;
	}
	REG_VALUE_ENTRY;
	int i;

	int RegBitName;
	unsigned long Value;

	static const REG_VALUE_ENTRY
AdditionalInitRegValueTable[RTL2832_TDA18272_ADDITIONAL_INIT_REG_TABLE_LEN] = {
		/* RegBitName,				Value*/
		{DVBT_DAGC_TRG_VAL,		0x39},
		{DVBT_AGC_TARG_VAL_0,		0x0},
		{DVBT_AGC_TARG_VAL_8_1,		0x32},
		{DVBT_AAGC_LOOP_GAIN,		0x16},
		{DVBT_LOOP_GAIN2_3_0,		0x6},
		{DVBT_LOOP_GAIN2_4,		0x1},
		{DVBT_LOOP_GAIN3,		0x16},
		{DVBT_VTOP1,			0x35},
		{DVBT_VTOP2,			0x21},
		{DVBT_VTOP3,			0x21},
		{DVBT_KRF1,			0x0},
		{DVBT_KRF2,			0x40},
		{DVBT_KRF3,			0x10},
		{DVBT_KRF4,			0x10},
		{DVBT_IF_AGC_MIN,		0x80},
		{DVBT_IF_AGC_MAX,		0x7f},
		{DVBT_RF_AGC_MIN,		0x80},
		{DVBT_RF_AGC_MAX,		0x7f},
		{DVBT_POLAR_RF_AGC,		0x0},
		{DVBT_POLAR_IF_AGC,		0x0},
		{DVBT_AD7_SETTING,		0xe9f4},
	};
		IfFreqHz      = IF_FREQ_4000000HZ;
		SpectrumMode  = SPECTRUM_INVERSE;



/* ==== Initialize DTMB demod and set its parameters =====*/

    /* Initialize demod.*/
    pDemod->Initialize(pDemod);


    /* Set demod parameters. (IF frequency and spectrum mode)*/


    pDemod->SetIfFreqHz(pDemod,      IfFreqHz);
    pDemod->SetSpectrumMode(pDemod,  SpectrumMode);



	/* Enable demod DVBT_IIC_REPEAT.*/
	if (pDemod->SetRegBitsWithPage(pDemod, DVBT_IIC_REPEAT, 0x1)
		 != FUNCTION_SUCCESS)
		goto error_status_set_registers;

	/* Disable demod DVBT_IIC_REPEAT.*/
	if (pDemod->SetRegBitsWithPage(pDemod, DVBT_IIC_REPEAT, 0x0)
		 != FUNCTION_SUCCESS)
		goto error_status_set_registers;

	/* Initialize demod.*/
	/* Note: TDA18272 tuner uses dynamic IF frequency,
	 so we will set demod IF frequency in SetParameters().*/
	if (pDemod->Initialize(pDemod) != FUNCTION_SUCCESS)
		goto error_status_execute_function;

	/* Set demod spectrum mode with SPECTRUM_INVERSE.*/
	if (pDemod->SetSpectrumMode(pDemod, SPECTRUM_INVERSE)
		 != FUNCTION_SUCCESS)
		goto error_status_execute_function;


	/* Set demod registers.*/
	for (i = 0; i < RTL2832_TDA18272_ADDITIONAL_INIT_REG_TABLE_LEN; i++) {
		/* Get register bit name and its value.*/
		RegBitName = AdditionalInitRegValueTable[i].RegBitName;
		Value      = AdditionalInitRegValueTable[i].Value;

		/* Set demod registers*/
		if (pDemod->SetRegBitsWithPage(pDemod, RegBitName, Value)
				 != FUNCTION_SUCCESS)
			goto error_status_set_registers;
	}

	return FUNCTION_SUCCESS;

error_status_execute_function:
error_status_set_registers:
	return FUNCTION_ERROR;
}
static int rtl2832_init(struct dvb_frontend *fe)
{
	struct rtl2832_state *state = fe->demodulator_priv;
	state->current_frequency = 0;
	printk(KERN_INFO " rtl2832_init().\n");
	rtl2832_init_demod(fe);

	printk(KERN_INFO " exit rtl2832_init().\n");

	return 0;
}

static int rtl2832_initial(struct dvb_frontend *fe)
{
	struct rtl2832_state *state = fe->demodulator_priv;
	unsigned char demod_i2c_address0 = 0x10;
	unsigned long demod_xtal_freq_hz = 28800000;
	state->current_frequency = 0;
	printk(KERN_INFO " rtl2832_initial().\n");

    /* Build base interface module.*/
	BuildBaseInterface(
	&(state->pBaseInterface),
	&(state->BaseInterfaceModuleMemory),
	9,			/* Set maximum I2C reading byte num with 9.*/
	8,			/* Set maximum I2C writing byte num with 8.*/
	CustomI2cRead,		/*  CustomI2cRead() as I2C reading function.*/
	CustomI2cWriteThenRead,	/*  CustomI2cRead() as I2C reading function.*/
	CustomI2cWrite,		/*  CustomI2cWrite() as I2C writing function.*/
	CustomWaitMs		/*  CustomWaitMs() as basic waiting function.*/
	);

	state->pBaseInterface->SetUserDefinedDataPointer(state->pBaseInterface
		 , state);

    /* Build DTMB demod  module.*/
	BuildRtl2832Module(
	&(state->pDemod),
	&(state->DtmbDemodModuleMemory),
	&(state->BaseInterfaceModuleMemory),
	&(state->I2cBridgeModuleMemory),
	demod_i2c_address0,	/*I2C device address is 0x3e in 8-bit format.*/
	demod_xtal_freq_hz,	/* Demod crystal frequency is 27.0 MHz.*/
	TS_INTERFACE_SERIAL	,/* Demod TS interface mode is serial.*/
	0,			/*AppMode*/
	100,			/*UpdateFuncRefPeriodMs*/
	0			/*IsFunc1Enabled*/
	);

	rtl2832_init_demod(fe);

	return 0;
}

static int rtl2832_read_status(struct dvb_frontend *fe, fe_status_t *status)
{
	struct rtl2832_state *state = fe->demodulator_priv;
	DVBT_DEMOD_MODULE *pDemod = state->pDemod;
	int answer;

	pDemod->IsSignalLocked(pDemod, &answer);
	if (answer)
		*status =  FE_HAS_LOCK | FE_HAS_SYNC
		 | FE_HAS_CARRIER | FE_HAS_SIGNAL;
	else
		*status = answer;

	return 1;
}

static int rtl2832_read_signal_strength(struct dvb_frontend *fe,
					u16 *signal_strength)
{
	struct rtl2832_state *state = fe->demodulator_priv;
	DVBT_DEMOD_MODULE *pDemod = state->pDemod;
	unsigned long SignalStrength;

	pDemod->GetSignalStrength(pDemod, &SignalStrength);
	*signal_strength = SignalStrength;

	return 0;
}

static int rtl2832_read_snr(struct dvb_frontend *fe, u16 *snr)
{
	struct rtl2832_state *state = fe->demodulator_priv;
	DVBT_DEMOD_MODULE *pDemod = state->pDemod;
	long SnrDbNum, SnrDbDen;

	pDemod->GetSnrDb(pDemod, &SnrDbNum, &SnrDbDen);
	*snr = SnrDbNum / SnrDbDen;

	return 0;
}

static int rtl2832_read_ucblocks(struct dvb_frontend *fe, u32 *ucblocks)
{
	struct rtl2832_state *state = fe->demodulator_priv;
	DVBT_DEMOD_MODULE *pDemod = state->pDemod;
	int answer;

	pDemod->IsSignalLocked(pDemod, &answer);
	*ucblocks = answer;

	return 0;
}

static int rtl2832_read_ber(struct dvb_frontend *fe, u32 *ber)
{
	struct rtl2832_state *state = fe->demodulator_priv;
	DVBT_DEMOD_MODULE *pDemod = state->pDemod;
	long BerNum, BerDen;

	pDemod->GetBer(pDemod, &BerNum, &BerDen);
	*ber = BerNum / BerDen;

	return 0;
}

static int rtl2832_get_frontend(struct dvb_frontend *fe,
				struct dvb_frontend_parameters *p)
{
	struct rtl2832_state *state = fe->demodulator_priv;

	p->frequency = state->current_frequency;
	p->u.vsb.modulation = state->current_modulation;

	return 0;
}

static int rtl2832_get_tune_settings(struct dvb_frontend *fe,
				     struct dvb_frontend_tune_settings *tune)
{
	return 0;
}

static void rtl2832_release(struct dvb_frontend *fe)
{
	struct rtl2832_state *state = fe->demodulator_priv;
	kfree(state);
}

static struct dvb_frontend_ops rtl2832_ops;

struct dvb_frontend *rtl2832_attach(const struct rtl2832_config *config,
				    struct i2c_adapter *i2c)
{
	struct rtl2832_state *state = NULL;

	printk(KERN_INFO " Enter rtl2832_attach(). attach success!\n");
	/* allocate memory for the internal state */
	state = kmalloc(sizeof(struct rtl2832_state), GFP_KERNEL);
	if (state == NULL)
		goto error;

	/* setup the state */
	state->config = config;
	state->i2c = i2c;
	state->current_modulation = QAM_16;
	state->inversion = state->config->inversion;

	/* create dvb_frontend */
	memcpy(&state->frontend.ops, &rtl2832_ops,
	       sizeof(struct dvb_frontend_ops));

	state->frontend.demodulator_priv = state;

	rtl2832_initial(&state->frontend);

	return &state->frontend;

error:
	kfree(state);
	return NULL;
}
EXPORT_SYMBOL(rtl2832_attach);

static struct dvb_frontend_ops rtl2832_ops = {

	.info = {
		 .name = "Realtek rtl2832 DVB-T Frontend",
		 .type = FE_OFDM,
		 .frequency_min = 177000000,
		 .frequency_max = 858000000,
		 .frequency_stepsize = 166666,
		 .caps = FE_CAN_FEC_1_2 | FE_CAN_FEC_2_3 | FE_CAN_FEC_3_4 |
		 FE_CAN_FEC_5_6 | FE_CAN_FEC_7_8 | FE_CAN_FEC_AUTO |
		 FE_CAN_QPSK | FE_CAN_QAM_16 | FE_CAN_QAM_64 | FE_CAN_QAM_AUTO |
		 FE_CAN_HIERARCHY_AUTO | FE_CAN_GUARD_INTERVAL_AUTO |
		 FE_CAN_TRANSMISSION_MODE_AUTO | FE_CAN_RECOVER},

	.init = rtl2832_init,
	.sleep = rtl2832_sleep,
	.set_frontend = rtl2832_set_frontend,
	.get_frontend = rtl2832_get_frontend,
	.get_tune_settings = rtl2832_get_tune_settings,
	.read_status = rtl2832_read_status,
	.read_ber = rtl2832_read_ber,
	.read_signal_strength = rtl2832_read_signal_strength,
	.read_snr = rtl2832_read_snr,
	.read_ucblocks = rtl2832_read_ucblocks,
	.release = rtl2832_release,
};

module_param(debug, int, 0644);
MODULE_PARM_DESC(debug, "Enable verbose debug messages");

MODULE_DESCRIPTION("Realtek rtl2832 DVB-T Demodulator driver");
MODULE_AUTHOR("Bill/Yi Liu <Bill.Liu@conexant.com>");
MODULE_LICENSE("GPL");
