/*
    tda18272-fe.c - driver for the Philips / NXP TDA18272 silicon tuner

    Copyright (C) 2011 Liuyi <Bill.Liu@conexant.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <linux/delay.h>
#include <linux/videodev2.h>
#include "tda18272-priv.h"
#include "tmNxTypes.h"
#include "tmddTDA182I2local.h"
#include "tmddTDA182I2Instance.h"
#define TMBSL_TDA18272
#ifdef TMBSL_TDA18272
 #include "tmbslTDA18272.h"
#else /* TMBSL_TDA18272 */
 #include "tmbslTDA18212.h"
#endif /* TMBSL_TDA18272 */

int tda18272_debug;
module_param_named(debug, tda18272_debug, int, 0644);
MODULE_PARM_DESC(debug, "set debug level "
		 "(info=1, map=2, reg=4, adv=8, cal=16 (or-able))");

MODULE_PARM_DESC(cal, "perform RF tracking filter calibration on startup");

static DEFINE_MUTEX(tda18272_list_mutex);
static LIST_HEAD(hybrid_tuner_instance_list);


static tmErrorCode_t   tda18272_i2c_read(tmUnitSelect_t tUnit, UInt32 AddrSize,
		 UInt8 *pAddr, UInt32 ReadLen, UInt8 *pData ,
		 void  *userPtr , int userData)
{
	struct dvb_frontend *fe = (struct dvb_frontend *)userPtr;

	struct tda18272_priv *priv = fe->tuner_priv;
	unsigned char buf = 0x00;
	unsigned char buf8[8];
	int ret = 0;
	unsigned char i = 0;

	struct i2c_msg msgwr[] = {
		{ .addr = priv->i2c_props.addr, .flags = 0,
		  .buf = &buf, .len = 1 },
		{ .addr = priv->i2c_props.addr, .flags = I2C_M_RD,
		  .buf = buf8, .len = 1 }
	};

	for (i = 0; i < ReadLen; i++) {
		buf = pAddr[0]+i;
		ret = i2c_transfer(priv->i2c_props.adap, msgwr, 2);
		pData[i] = buf8[0];
	}

	if (ret != 2)
		tda_err("ERROR: i2c_transfer returned: %d\n", ret);

   return TM_OK;
}

static tmErrorCode_t tda18272_i2c_write(tmUnitSelect_t tUnit, UInt32 AddrSize,
			 UInt8 *pAddr, UInt32 WriteLen, UInt8 *pData ,
			 void  *userPtr , int userData)
{
struct dvb_frontend *fe = (struct dvb_frontend *)userPtr;

	struct tda18272_priv *priv = fe->tuner_priv;
	unsigned char buf8[2];
	int ret = 0;
	unsigned char i = 0;

	struct i2c_msg msgwrite[] = {
		{ .addr = priv->i2c_props.addr, .flags = 0,
		  .buf = buf8, .len = 2 }
	};

	for (i = 0; i < WriteLen; i++) {
		buf8[0] = pAddr[0]+i;
		buf8[1] = pData[i];
		ret = i2c_transfer(priv->i2c_props.adap, msgwrite, 1);
	}

    return TM_OK;
}

static tmErrorCode_t   tda18272_wait(tmUnitSelect_t tUnit, UInt32 tms)
{
	msleep(tms);
	return TM_OK;
}


static tmErrorCode_t   tda18272_print(UInt32 level, const char *format, ...)
{
	char pszDest[200];
	int DestLen = 199;
	int ret = 0;
	va_list arglist;

	va_start(arglist, format);
	ret = vsnprintf(pszDest, DestLen, format, arglist);
	printk(KERN_INFO "%s\n", pszDest);
	va_end(arglist);

	return TM_OK;
}
/*---------------------------------------------------------------------*/

static int tda18272_initialize(struct dvb_frontend *fe)
{
	u32 frequency = 0;
	tmUnitSelect_t unit_sel = 0;
	tmbslFrontEndDependency_t fed;

	tda_info("tda18272_initialize()\n");

	memset(&fed , 0, sizeof(fed));
	fed.sTime.Wait = tda18272_wait;
	fed.sDebug.Print = tda18272_print;
	fed.sIo.Read = tda18272_i2c_read;
	fed.sIo.Write = tda18272_i2c_write;
	fed.sIo.userPtr = fe;

	frequency = 0;
	tmbslTDA182I2Init(unit_sel , &fed);
	tmbslTDA182I2Reset(unit_sel);
	tmbslTDA182I2SetStandardMode(unit_sel, tmTDA182I2_ANLG_MN);
	tmbslTDA182I2SetRf(unit_sel , frequency);

	return TM_OK;
}

static int tda18272_init(struct dvb_frontend *fe)
{
	int ret = TM_OK;

	tda_info("tda18272_init()\n");

	return ret;
}

static int tda18272_sleep(struct dvb_frontend *fe)
{
	int ret = TM_OK;

	return ret;
}



/* ------------------------------------------------------------------ */

static int tda18272_set_params(struct dvb_frontend *fe,
			       struct dvb_frontend_parameters *params)
{
	struct tda18272_priv *priv = fe->tuner_priv;
	struct tda18272_std_map *std_map = &priv->std;
	struct tda18272_std_map_item *map;
	int ret;
	u32 bw, freq = params->frequency;
	tmTDA182I2StandardMode_t standard = tmTDA182I2_DVBT_8MHz;

	printk(KERN_INFO "tda18272_set_params!\n");
	priv->mode = TDA18272_DIGITAL;

	if (fe->ops.info.type == FE_ATSC) {
		switch (params->u.vsb.modulation) {
		case VSB_8:
		case VSB_16:
			map = &std_map->atsc_6;
			standard = tmTDA182I2_ATSC_6MHz;
			break;
		case QAM_64:
		case QAM_256:
			map = &std_map->qam_6;
			standard = tmTDA182I2_QAM_6MHz;
			break;
		default:
			standard = tmTDA182I2_ATSC_6MHz;
			tda_warn("modulation not set!\n");
			/*return -EINVAL;*/
		}
#if 0
		/* userspace request is already center adjusted */
		freq += 1750000; /* Adjust to center (+1.75MHZ) */
#endif
		bw = 0;
	} else if (fe->ops.info.type == FE_OFDM) {
		switch (params->u.ofdm.bandwidth) {
		case BANDWIDTH_6_MHZ:
			standard = tmTDA182I2_DVBT_6MHz;
			map = &std_map->dvbt_6;
			break;
		case BANDWIDTH_7_MHZ:
			standard = tmTDA182I2_DVBT_7MHz;
			map = &std_map->dvbt_7;
			break;
		case BANDWIDTH_8_MHZ:
			standard = tmTDA182I2_DVBT_8MHz;
			map = &std_map->dvbt_8;
			break;
		default:
			tda_warn("bandwidth not set!\n");
			return -EINVAL;
		}
	} else {
		tda_warn("modulation type not supported!\n");
		return -EINVAL;
	}

	/* When tuning digital, the analog demod must be tri-stated */
	if (fe->ops.analog_ops.standby)
		fe->ops.analog_ops.standby(fe);

	printk(KERN_INFO "$$$ tda18272_set_params!!!standard=%d;freq=%d\n",
		standard, freq);
	mutex_lock(&priv->lock);
	ret = (tmbslTDA182I2SetStandardMode(0 , standard) == TM_OK ? 1 : 0);
	ret = tmbslTDA182I2SetRf(0 , freq) == TM_OK ? 1 : 0;
	mutex_unlock(&priv->lock);
	if (tda_fail(ret))
		goto fail;

	priv->frequency = freq;
	priv->bandwidth = (fe->ops.info.type == FE_OFDM) ?
		params->u.ofdm.bandwidth : 0;
fail:
	return ret;

}

static int tda18272_set_analog_params(struct dvb_frontend *fe,
				      struct analog_parameters *params)
{
	struct tda18272_priv *priv = fe->tuner_priv;
	int ret = 0;
	u32 standard_mode = 0;
	tmUnitSelect_t unit_sel = 0;
	u32 freq = params->frequency * 125 *
		((params->mode == V4L2_TUNER_RADIO) ? 1 : 1000) / 2;

	tda_info("tda18272_set_analog_params() freq=%d params->mode=0x%x\n",
			freq, params->mode);

	if (params->mode == V4L2_TUNER_RADIO)
		standard_mode = tmTDA182I2_FM_Radio;
	else if (params->std & V4L2_STD_MN)
		standard_mode = tmTDA182I2_ANLG_MN;
	else if (params->std & V4L2_STD_B)
		standard_mode = tmTDA182I2_ANLG_B;
	else if (params->std & V4L2_STD_GH)
		standard_mode = tmTDA182I2_ANLG_GH;
	else if (params->std & V4L2_STD_PAL_I)
		standard_mode = tmTDA182I2_ANLG_I;
	else if (params->std & V4L2_STD_DK)
		standard_mode = tmTDA182I2_ANLG_DK;
	else if (params->std & V4L2_STD_SECAM_L)
		standard_mode = tmTDA182I2_ANLG_L;
	else if (params->std & V4L2_STD_SECAM_LC)
		standard_mode = tmTDA182I2_ANLG_LL;
	else
		standard_mode = tmTDA182I2_ANLG_MN;


	mutex_lock(&priv->lock);
	ret = (tmbslTDA182I2SetStandardMode(unit_sel ,
		 standard_mode) == TM_OK ? 1 : 0);
	ret = tmbslTDA182I2SetRf(unit_sel , freq) == TM_OK ? 1 : 0;
	mutex_unlock(&priv->lock);

    return ret;
}

static int tda18272_release(struct dvb_frontend *fe)
{
	struct tda18272_priv *priv = fe->tuner_priv;

	mutex_lock(&tda18272_list_mutex);
	tmbslTDA182I2DeInit(0);
	if (priv)
		hybrid_tuner_release_state(priv);

	mutex_unlock(&tda18272_list_mutex);

	fe->tuner_priv = NULL;

	return 0;
}

static int tda18272_get_frequency(struct dvb_frontend *fe, u32 *frequency)
{
	struct tda18272_priv *priv = fe->tuner_priv;
	*frequency = priv->frequency;
	return 0;
}

static int tda18272_get_bandwidth(struct dvb_frontend *fe, u32 *bandwidth)
{
	struct tda18272_priv *priv = fe->tuner_priv;
	*bandwidth = priv->bandwidth;
	return 0;
}

static int tda18272_set_config(struct dvb_frontend *fe, void *priv_cfg)
{
	return 0;
}

int tda18272_read_regs(struct dvb_frontend *fe)
{
	struct tda18272_priv *priv = fe->tuner_priv;
	unsigned char *regs = priv->tda18272_regs;
	unsigned char buf = 0x00;
	int ret;

	struct i2c_msg msg1[] = {
		{ .addr = priv->i2c_props.addr, .flags = 0,
		  .buf = &buf, .len = 1 }
	};
	struct i2c_msg msg[] = {
		{ .addr = priv->i2c_props.addr, .flags = I2C_M_RD,
		  .buf = regs, .len = 20 }
	};


	/* read all registers */
	ret = i2c_transfer(priv->i2c_props.adap, msg1, 1);
	ret = i2c_transfer(priv->i2c_props.adap, msg, 1);

	if (ret != 2)
		tda_err("ERROR: i2c_transfer returned: %d\n", ret);

	return (ret == 2 ? 0 : ret);
}
static int tda18272_get_id(struct dvb_frontend *fe)
{
	struct tda18272_priv *priv = fe->tuner_priv;
	unsigned char *regs = priv->tda18272_regs;
	char *name;

	mutex_lock(&priv->lock);
	tda18272_read_regs(fe);
	mutex_unlock(&priv->lock);

	if ((regs[R_ID] == 0xC7) && (regs[R_TM] == 0x60)) {
		name = "TDA18272HN/M";
		priv->id = TDA18272HNM;
	} else {
		tda_info("Unknown device (%i) detected @ %d-%04x, device not supported.\n",
			 regs[R_ID], i2c_adapter_id(priv->i2c_props.adap),
			 priv->i2c_props.addr);
		return -EINVAL;
	}

	tda_info("%s detected @ %d-%04x\n", name,
		 i2c_adapter_id(priv->i2c_props.adap), priv->i2c_props.addr);

	return 0;
}

static int tda18272_setup_configuration(struct dvb_frontend *fe,
					struct tda18272_config *cfg)
{
	struct tda18272_priv *priv = fe->tuner_priv;

	priv->gate = (cfg) ? cfg->gate : TDA18272_GATE_AUTO;
	priv->role = (cfg) ? cfg->role : TDA18272_MASTER;
	priv->config = (cfg) ? cfg->config : 0;
	priv->small_i2c = (cfg) ?
		cfg->small_i2c : TDA18272_39_BYTE_CHUNK_INIT;
	priv->output_opt = (cfg) ?
		cfg->output_opt : TDA18272_OUTPUT_LT_XT_ON;

	return 0;
}

static int tda18272_update_std_map(struct dvb_frontend *fe,
				   struct tda18272_std_map *map)
{
	if (!map)
		return -EINVAL;

	return 0;
}
static struct dvb_tuner_ops tda18272_tuner_ops = {
	.info = {
		.name = "NXP TDA18272HN/M",
		.frequency_min  =  45000000,
		.frequency_max  = 864000000,
		.frequency_step =     62500
	},
	.init              = tda18272_init,
	.sleep             = tda18272_sleep,
	.set_params        = tda18272_set_params,
	.set_analog_params = tda18272_set_analog_params,
	.release           = tda18272_release,
	.set_config        = tda18272_set_config,
	.get_frequency     = tda18272_get_frequency,
	.get_bandwidth     = tda18272_get_bandwidth,
};

struct dvb_frontend *tda18272_attach(struct dvb_frontend *fe, u8 addr,
				     struct i2c_adapter *i2c,
				     struct tda18272_config *cfg)
{
	struct tda18272_priv *priv = NULL;
	int instance, ret;
	printk(KERN_INFO "tda18272_attach!\n");
	mutex_lock(&tda18272_list_mutex);

	instance = hybrid_tuner_request_state(struct tda18272_priv, priv,
					      hybrid_tuner_instance_list,
					      i2c, addr, "tda18272");
	switch (instance) {
	case 0:
		goto fail;
	case 1:
		/* new tuner instance */
		fe->tuner_priv = priv;
		tda18272_setup_configuration(fe, cfg);

		priv->cal_initialized = false;
		mutex_init(&priv->lock);

		ret = tda18272_get_id(fe);
		if (tda_fail(ret))
			goto fail;

		mutex_lock(&priv->lock);

		tda18272_initialize(fe);

		mutex_unlock(&priv->lock);
		break;
	default:
		/* existing tuner instance */
		fe->tuner_priv = priv;

		/* allow dvb driver to override configuration settings */
		if (cfg) {
			if (cfg->gate != TDA18272_GATE_ANALOG)
				priv->gate = cfg->gate;
			if (cfg->role)
				priv->role = cfg->role;
			if (cfg->config)
				priv->config = cfg->config;
			if (cfg->small_i2c)
				priv->small_i2c = cfg->small_i2c;
			if (cfg->output_opt)
				priv->output_opt = cfg->output_opt;
			if (cfg->std_map)
				tda18272_update_std_map(fe, cfg->std_map);
		}
			tda18272_init(fe);
		break;
	}

	/* override default std map with values in config struct */
	if ((cfg) && (cfg->std_map))
		tda18272_update_std_map(fe, cfg->std_map);

	mutex_unlock(&tda18272_list_mutex);

	memcpy(&fe->ops.tuner_ops, &tda18272_tuner_ops,
	       sizeof(struct dvb_tuner_ops));


	return fe;
fail:
	printk(KERN_INFO "tda18272_attach fail!\n");
	mutex_unlock(&tda18272_list_mutex);

	tda18272_release(fe);
	return NULL;
}
EXPORT_SYMBOL_GPL(tda18272_attach);
MODULE_DESCRIPTION("NXP TDA18272HN/M analog / digital tuner driver");
MODULE_AUTHOR("Bill/Yi Liu <Bill.Liu@conexant.com>");
MODULE_LICENSE("GPL");
MODULE_VERSION("0.4");

/*
 * Overrides for Emacs so that we follow Linus's tabbing style.
 * ---------------------------------------------------------------------------
 * Local variables:
 * c-basic-offset: 8
 * End:
 */
