#ifndef __DEMOD_RTD2885_ATSC_H
#define __DEMOD_RTD2885_ATSC_H

/**

@file

@brief   RTD2885 ATSC demod module declaration

One can manipulate RTD2885 ATSC demod through RTD2885 ATSC module.
RTD2885 ATSC module is derived from ATSC demod module.



@par Example:
@code

// The example is the same as the ATSC demod example in atsc_demod_base.h except the listed lines.



#include "demod_rtd2885_atsc.h"


...



int main(void)
{
    ATSC_DEMOD_MODULE *pDemod;

    ATSC_DEMOD_MODULE     AtscDemodModuleMemory;
    BASE_INTERFACE_MODULE BaseInterfaceModuleMemory;
    I2C_BRIDGE_MODULE     I2cBridgeModuleMemory;


    ...



    // Build RTD2885 ATSC demod module.
    BuildRtd2885AtscModule(
        &pDemod,
        &AtscDemodModuleMemory,
        &BaseInterfaceModuleMemory,
        &I2cBridgeModuleMemory,
        0x24,                               // I2C device address is 0x24 in 8-bit format.
        CRYSTAL_FREQ_27000000HZ,            // Crystal frequency is 27 MHz.
        TS_INTERFACE_SERIAL                 // TS interface mode is serial.
        );



    // See the example for other ATSC demod functions in atsc_demod_base.h

    ...


    return 0;
}


@endcode

*/


#include "atsc_demod_base.h"





// Definitions
#define RTD2885_ATSC_REG_TABLE_LEN              61
#define RTD2885_ATSC_SYS_VERSION_VALUE          0xa3



// AGC register
#define RTD2885_ATSC_RF_AGC_VALUE_BIT_NUM       14
#define RTD2885_ATSC_IF_AGC_VALUE_BIT_NUM       14

// TR offset register
#define RTD2885_ATSC_TR_OFFSET_N1_BIT_NUM       11
#define RTD2885_ATSC_TR_OFFSET_N2_BIT_NUM       16

// CR offset register
#define RTD2885_ATSC_CR_OFFSET_N1_BIT_NUM               18
#define RTD2885_ATSC_CR_OFFSET_N2_25_0_BIT_NUM          26
#define RTD2885_ATSC_CR_OFFSET_N2_32_26_BIT_NUM         7

// BER and UPER
#define RTD2885_ATSC_BER_WAIT_TIME_MS           10
#define RTD2885_ATSC_BER_REG2_MSB_SHIFT         16

// SNR
// Note: RTD2885_ATSC_SNR_DB_NUM_CONST = round(10 * log2(21 * pow(2, 26)) * pow(2, RTD2885_ATSC_SNR_FRAC_BIT_NUM))
//       RTD2885_ATSC_SNR_DB_DEN       = round(log2(10) * pow(2, RTD2885_ATSC_SNR_FRAC_BIT_NUM))
#define RTD2885_ATSC_SNR_FRAC_BIT_NUM           10
#define RTD2885_ATSC_SNR_DB_NUM_CONST           311217
#define RTD2885_ATSC_SNR_DB_DEN                 3402
#define RTD2885_ATSC_MSE2_23_0_BIT_NUM          24
#define RTD2885_ATSC_MSE2_33_24_BIT_NUM         10





// Table length
#define RTD2885_ATSC_INIT_REG_TABLE_LEN                     590
#define RTD2885_ATSC_TS_INTERFACE_INIT_TABLE_LEN            3





// Builder
void
BuildRtd2885AtscModule(
    ATSC_DEMOD_MODULE **ppDemod,
    ATSC_DEMOD_MODULE *pAtscDemodModuleMemory,
    BASE_INTERFACE_MODULE *pBaseInterfaceModuleMemory,
    I2C_BRIDGE_MODULE *pI2cBridgeModuleMemory,
    unsigned char DeviceAddr,
    unsigned long CrystalFreqHz,
    int TsInterfaceMode
    );





// Manipulaing functions
void
rtd2885_atsc_IsConnectedToI2c(
    ATSC_DEMOD_MODULE *pDemod,
    int *pAnswer
    );

int
rtd2885_atsc_SoftwareReset(
    ATSC_DEMOD_MODULE *pDemod
    );

int
rtd2885_atsc_Initialize(
    ATSC_DEMOD_MODULE *pDemod
    );

int
rtd2885_atsc_SetSymbolRateHz(
    ATSC_DEMOD_MODULE *pDemod,
    unsigned long SymbolRateHz
    );

int
rtd2885_atsc_SetIfFreqHz(
    ATSC_DEMOD_MODULE *pDemod,
    unsigned long IfFreqHz
    );

int
rtd2885_atsc_SetSpectrumMode(
    ATSC_DEMOD_MODULE *pDemod,
    int SpectrumMode
    );

int
rtd2885_atsc_GetRfAgc(
    ATSC_DEMOD_MODULE *pDemod,
    long *pRfAgc
    );

int
rtd2885_atsc_GetIfAgc(
    ATSC_DEMOD_MODULE *pDemod,
    long *pIfAgc
    );

int
rtd2885_atsc_GetDiAgc(
    ATSC_DEMOD_MODULE *pDemod,
    unsigned long *pDiAgc
    );

int
rtd2885_atsc_GetTrOffsetPpm(
    ATSC_DEMOD_MODULE *pDemod,
    long *pTrOffsetPpm
    );

int
rtd2885_atsc_GetCrOffsetHz(
    ATSC_DEMOD_MODULE *pDemod,
    long *pCrOffsetHz
    );

int
rtd2885_atsc_IsAagcLocked(
    ATSC_DEMOD_MODULE *pDemod,
    int *pAnswer
    );

int
rtd2885_atsc_IsFrameLocked(
    ATSC_DEMOD_MODULE *pDemod,
    int *pAnswer
    );

int
rtd2885_atsc_GetErrorRate(
    ATSC_DEMOD_MODULE *pDemod,
    unsigned long TestVolume,
    unsigned int WaitTimeMsMax,
    unsigned long *pByteErrorRateNum,
    unsigned long *pByteErrorRateDen,
    unsigned long *pPacketErrorRateNum,
    unsigned long *pPacketErrorRateDen
    );

int
rtd2885_atsc_GetSnrDb(
    ATSC_DEMOD_MODULE *pDemod,
    long *pSnrDbNum,
    long *pSnrDbDen
    );

int
rtd2885_atsc_GetSignalStrength(
    ATSC_DEMOD_MODULE *pDemod,
    unsigned long *pSignalStrength
    );

int
rtd2885_atsc_GetSignalQuality(
    ATSC_DEMOD_MODULE *pDemod,
    unsigned long *pSignalQuality
    );

int
rtd2885_atsc_UpdateFunction(
    ATSC_DEMOD_MODULE *pDemod
    );

int
rtd2885_atsc_ResetFunction(
    ATSC_DEMOD_MODULE *pDemod
    );





// I2C command forwarding functions
int
rtd2885_atsc_ForwardI2cReadingCmd(
    I2C_BRIDGE_MODULE *pI2cBridge,
    unsigned char DeviceAddr,
    unsigned char *pReadingBytes,
    unsigned long ByteNum
    );

int
rtd2885_atsc_ForwardI2cWritingCmd(
    I2C_BRIDGE_MODULE *pI2cBridge,
    unsigned char DeviceAddr,
    const unsigned char *pWritingBytes,
    unsigned long ByteNum
    );





// Register table initialization
void
rtd2885_atsc_InitRegTable(
    ATSC_DEMOD_MODULE *pDemod
    );





// I2C birdge module builder
void
rtd2885_atsc_BuildI2cBridgeModule(
    ATSC_DEMOD_MODULE *pDemod
    );















#endif
